/*--------------------------------------------------------------------------+
$Id: ModelWriter.java 26268 2010-02-18 10:44:30Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.xml.example;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintStream;

import edu.tum.cs.commons.xml.XMLWriter;

/**
 * Example model writer.
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * @version $Rev: 26268 $
 * @levd.rating GREEN Hash: 935B5C0096AFC56DC02E322CFEBFFDF7
 */
public class ModelWriter extends XMLWriter<EXMLElement, EXMLAttribute> {

	/** Model to write. */
	private final Model model;

	/**
	 * Create new model writer.
	 * 
	 * @param file
	 *            the file to write to
	 * @param model
	 *            the model to persist
	 * @throws FileNotFoundException
	 *             if file could not be created.
	 */
	public ModelWriter(File file, Model model) throws FileNotFoundException {
		super(new PrintStream(file), new ModelResolver());
		this.model = model;
	}

	/** Write model. */
	public void writeModel() {
		// add XML header with version and encoding information
		addHeader("1.0", "ISO-8859-1");

		// start model root element
		openElement(EXMLElement.MODEL);

		// write persons
		for (Person person : model.getPersons()) {
			write(person);
		}

		// close root element
		closeElement(EXMLElement.MODEL);

		// close underlying stram
		close();
	}

	/**
	 * Write address.
	 */
	private void write(Address address) {
		openElement(EXMLElement.ADDRESS);

		openElement(EXMLElement.STREET);
		addText(address.getStreet());
		closeElement(EXMLElement.STREET);

		openElement(EXMLElement.CITY);
		addText(address.getCity());
		closeElement(EXMLElement.CITY);

		closeElement(EXMLElement.ADDRESS);
	}

	/** Write order. */
	private void write(Order order) {
		// start element
		openElement(EXMLElement.ORDER);

		// add attribute
		addAttribute(EXMLAttribute.ARTICLE_NUMBER, order.getArticleNumber());

		addText(order.getDescription());

		closeElement(EXMLElement.ORDER);
	}

	/**
	 * Write a person.
	 */
	private void write(Person person) {
		openElement(EXMLElement.PERSON);

		// write name as text content of element NAME
		openElement(EXMLElement.NAME);
		addText(person.getName());
		closeElement(EXMLElement.NAME);

		// write address
		write(person.getAddress());
		
		openElement(EXMLElement.DETAILS);
		addCDataSection(person.getDetails());
		closeElement(EXMLElement.DETAILS);

		openElement(EXMLElement.ORDERS);

		// write orders
		for (Order order : person.getOrders()) {
			write(order);
		}

		closeElement(EXMLElement.ORDERS);

		closeElement(EXMLElement.PERSON);

	}

}