/*
 * TicTacToe for Palm Pilot
 * Copyright (C) 1998  Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>

#include "tictactoe.h"
#include "circle.h"

/*
 * Game board looks like this:
 *
 *   0 | 1 | 2
 *  ---+---+---
 *   3 | 4 | 5
 *  ---+---+---
 *   6 | 7 | 8
 */

#define TicTacToeCreator 'TTT9'

#define PosState_Empty 2
#define PosState_X     3
#define PosState_O     5

#define PlayerType_Human    0
#define PlayerType_Computer 1

#define maxRows 8
#define maxCols 8

typedef struct {
    int numRows;
    int numCols;
    int pos[maxRows * maxCols];
    int neededToWin;
    int playerType[2];
    int currentPlayer;
    int gameover;
    int turn;
} BoardType;

typedef BoardType * BoardPtr;

static BoardType board;
static Boolean boardIsOnScreen;

static SWord BoardX, BoardY, PieceWidth, PieceHeight;

static void
InitBoard(BoardPtr board, int numRows, int numCols)
{
    int i;

    board->numRows = numRows;
    board->numCols = numCols;
    for (i = 0; i < numRows * numCols; i++)
	board->pos[i] = PosState_Empty;
    board->neededToWin = numCols > numRows ? numCols : numRows;
    board->playerType[0] = PlayerType_Human;
    board->playerType[1] = PlayerType_Computer;
    board->currentPlayer = 0;
    board->turn = 0;
    board->gameover = 0;
}

static void
NewBoard(BoardPtr board)
{
    int i;

    for (i = 0; i < 9; i++) board->pos[i] = PosState_Empty;
    board->currentPlayer = 0;
    board->gameover = 0;
    board->turn = 0;
}

static void
GetPieceInfo(int row, int col, RectanglePtr r, int * i)
{
    r->topLeft.x = BoardX + col * (PieceWidth+1);
    r->topLeft.y = BoardY + row * (PieceHeight+1);
    r->extent.x = PieceWidth;
    r->extent.y = PieceHeight;
    *i = (row * board.numRows) + col;
}

static void
DrawPiece(RectanglePtr r, int state)
{
    RectangleType r1;
    SWord x1,x2,y1,y2;

    switch (state) {
    case PosState_Empty:
	WinEraseRectangle(r, 0);
	break;

    case PosState_X:
	x1 = r->topLeft.x + 2;
	y1 = r->topLeft.y + 2;
	x2 = r->topLeft.x + r->extent.x - 3;
	y2 = r->topLeft.y + r->extent.y - 3;

	WinDrawLine(x1,y1,x2,y2);
	WinDrawLine(x1+1,y1,x2,y2-1);
	WinDrawLine(x1,y1+1,x2-1,y2);

	WinDrawLine(x2,y1,x1,y2);
	WinDrawLine(x2-1,y1,x1,y2-1);
	WinDrawLine(x2,y1+1,x1+1,y2);
    
	break;

    case PosState_O:
	WinDrawCircle(r->topLeft.x + r->extent.x / 2,
		      r->topLeft.y + r->extent.y / 2,
		      (r->extent.x - 4) / 2);
	WinDrawCircle(r->topLeft.x + r->extent.x / 2,
		      r->topLeft.y + r->extent.y / 2,
		      ((r->extent.x - 4) / 2) - 1);

	break;
    }
}

static void
DrawBoard(FormPtr form)
{
    RectangleType r;
    int row,col;

    /* Draw the tic-tac-toe frames. */
    for (col = 1; col < board.numCols; col++)
	WinDrawLine(BoardX + col*PieceWidth + (col-1),
		    BoardY,
		    BoardX + col*PieceWidth + (col-1),
		    BoardY + board.numRows*PieceHeight + board.numRows-1);

    for (row = 1; row < board.numRows; row++)
	WinDrawLine(BoardX,
		    BoardY + row*PieceHeight + (row-1),
		    BoardX+board.numCols*PieceWidth + (board.numCols-1),
		    BoardY + row*PieceHeight + (row-1));

    /* Draw each piece on the board. */
    for (row = 0; row < board.numRows; row++)
	for (col = 0; col < board.numCols; col++) {
	    int i;

	    GetPieceInfo(row, col, &r, &i);
	    DrawPiece(&r, board.pos[i]);
	}

    boardIsOnScreen = true;
}

static int
CheckWin(void)
{
    int row,col,p1,p2,p,i;

    /* Calculate the products we want to look for. */
    p1 = p2 = 1;
    for (i = 0; i < board.neededToWin; i++) {
	p1 *= PosState_X;
	p2 *= PosState_O;
    }

    /* See if the game has been won horizontally. */
    for (row = 0; row < board.numRows; row++) {
	int index = row * board.numRows;
	for (col = 0; col <= board.numCols - board.neededToWin; col++) {
	    p = 1;
	    for (i = col; i < col + board.neededToWin; i++)
		p *= board.pos[index + i];
	    if (p == p1 || p == p2)
		return 1;
	}
    }

    /* See if the game was been won vertically. */
    for (col = 0; col < board.numCols; col++) {
	for (row = 0; row <= board.numRows - board.neededToWin; row++) {
	    int index = row * board.numRows + col;
	    p = 1;
	    for (i = 0; i < board.neededToWin; i++)
		p *= board.pos[index + i * board.numCols];
	    if (p == p1 || p == p2)
		return 1;
	}
    }

    /* See if the games has been won on a downward diagonal. */
    for (row = 0; row <= board.numRows - board.neededToWin; row++)
	for (col = 0; col <= board.numCols - board.neededToWin; col++) {
	    p = 1;
	    for (i = 0; i < board.neededToWin; i++)
		p *= board.pos[((row+i)*board.numRows)+(col+i)];
	    if (p == p1 || p == p2)
		return 1;
	}

    /* See if the game has been won on an upward diagonal. */
    for (row = 0; row <= board.numRows - board.neededToWin; row++)
	for (col = board.numCols - 1; col >= board.neededToWin - 1; col--) {
	    p = 1;
	    for (i = 0; i < board.neededToWin; i++)
		p *= board.pos[(row+i)*board.numRows+(col-i)];
	    if (p == p1 || p == p2)
		return 1;
	}

    return 0;
}

static void
MakeMove(int row, int col)
{
    RectangleType r;
    int i, piece;

    GetPieceInfo(row, col, &r, &i);
    if (board.pos[i] == PosState_Empty) {
	piece = (board.currentPlayer == 0) ? PosState_X : PosState_O;
	board.pos[i] = piece;
	DrawPiece(&r, piece);
	board.turn++;
	if (board.turn >= board.numCols*board.numRows || CheckWin())
	    board.gameover = 1;
	board.currentPlayer = (board.currentPlayer == 1) ? 0 : 1;
    }
}

/*
 * Computer Movements.
 */

static Boolean
PossibleWin(int piece, int * rowP, int * colP)
{
    int row,col,i,product,p;

    /* Calculate the product we want to look for. */
    product = PosState_Empty;
    for (i = 0; i < board.neededToWin - 1; i++)
	product *= piece;

    /* See if we can win horizontally. */
    for (row = 0; row < board.numRows; row++) {
	int index = row * board.numRows;
	for (col = 0; col <= board.numCols - board.neededToWin; col++) {
	    p = 1;
	    for (i = col; i < col + board.neededToWin; i++)
		p *= board.pos[index + i];
	    if (p == product) {
		for (i = col; i < col + board.neededToWin; i++)
		    if (board.pos[index + i] == PosState_Empty) {
			*rowP = row;
			*colP = i;
			return true;
		    }
	    }
	}
    }

    /* See if we can win vertically. */
    for (col = 0; col < board.numCols; col++) {
	for (row = 0; row <= board.numRows - board.neededToWin; row++) {
	    int index = row * board.numRows + col;
	    p = 1;
	    for (i = 0; i < board.neededToWin; i++)
		p *= board.pos[index + i * board.numCols];
	    if (p == product) {
		for (i = row; i < row + board.neededToWin; i++)
		    if (board.pos[col + i * board.numCols] == PosState_Empty){
			*rowP = i;
			*colP = col;
			return true;
		    }
	    }
	}
    }

    /* See if we can win on a downward diagonal. */
    for (row = 0; row <= board.numRows - board.neededToWin; row++)
	for (col = 0; col <= board.numCols - board.neededToWin; col++) {
	    p = 1;
	    for (i = 0; i < board.neededToWin; i++)
		p *= board.pos[(row+i)*board.numRows+(col+i)];
	    if (p == product) {
		for (i = 0; i < board.neededToWin; i++)
		    if (board.pos[(row+i)*board.numRows+(col+i)]
			== PosState_Empty) {
			*rowP = row + i;
			*colP = col + i;
			return true;
		    }
	    }
	}

    /* See if we can win on an upward diagonal. */
    for (row = 0; row <= board.numRows - board.neededToWin; row++)
	for (col = board.numCols - 1; col >= board.neededToWin - 1; col--) {
	    p = 1;
	    for (i = 0; i < board.neededToWin; i++)
		p *= board.pos[(row+i)*board.numRows+(col-i)];
	    if (p == product) {
		for (i = 0; i < board.neededToWin; i++)
		    if (board.pos[(row+i)*board.numRows+(col-i)]
			== PosState_Empty) {
			*rowP = row + i;
			*colP = col - i;
			return true;
		    }
	    }
	}

    return false;
}

static void
DoComputerMove(void)
{
    int row, col, ourPiece;

    ourPiece = (board.currentPlayer == 0) ? PosState_X : PosState_O;

    /* Try to win the game. */
    if (PossibleWin(ourPiece, &row, &col)) {
	MakeMove(row,col);
	return;
    }

    /* Try to stop the opponent from winning the game. */
    if (PossibleWin(ourPiece == PosState_X ? PosState_O : PosState_X,
		    &row, &col)) {
	MakeMove(row,col);
	return;
    }

    /* Otherwise, try a normal move. */
    for (row = 0; row < board.numRows; row++)
	for (col = 0; col < board.numCols; col++)
	  if (board.pos[row*board.numRows+col] == PosState_Empty) {
	      MakeMove(row,col);
	      return;
	  }
}

static Boolean
MakeComputerMove(void)
{
    if (board.playerType[board.currentPlayer] == PlayerType_Computer) {
	DoComputerMove();
    }
}

static Boolean
MapPenPosition(SWord x, SWord y, int * row, int * col)
{
    RectangleType r;

    WinDisplayToWindowPt(&x, &y);

    r.topLeft.x = BoardX;
    r.topLeft.y = BoardY;
    r.extent.x = (board.numCols * (PieceWidth + 1)) - 1;
    r.extent.y = (board.numRows * (PieceHeight + 1)) - 1;

    if (! RctPtInRectangle(x, y, &r))
	return false;

    x -= BoardX;
    y -= BoardY;

    *row = y / (PieceWidth  + 1);
    *col = x / (PieceHeight + 1);

    return true;
}

static Boolean
BoardPenDown(SWord X, SWord Y)
{
    int row, col;

    if (board.gameover)
	return false;

    if (MapPenPosition(X, Y, &row, &col)) {
	MakeMove(row, col);
	return true;
    }

    return false;
}

static void
NewGame(void)
{
    NewBoard(&board);
    DrawBoard(FrmGetActiveForm());
}

static Boolean
BoardHandleEvent(EventPtr event)
{
    FormPtr form;

    switch (event->eType) {
    case frmOpenEvent:
	BoardX = 20;
	BoardY = 20;
	PieceWidth = (120 - (board.numCols - 1)) / board.numCols;
	PieceHeight = (120 - (board.numRows - 1)) / board.numRows;

	form = FrmGetActiveForm();
	FrmDrawForm(form);
	DrawBoard(form);
	return true;

    case frmUpdateEvent:
	form = FrmGetActiveForm();
	FrmDrawForm(form);
	DrawBoard(form);
	return true;

    case penDownEvent:
	return BoardPenDown(event->screenX, event->screenY);

    case ctlSelectEvent:
	if (event->data.ctlEnter.controlID == ctlID_NewButton) {
	    NewGame();
	    return true;
	}
	break;

    case menuEvent:
	switch (event->data.menu.itemID) {
	case menuitemID_new:
	    NewGame();
	    break;

	case menuitemID_pref:
	    FrmPopupForm(formID_Preferences);
	    break;

	case menuitemID_about:
	    FrmAlert(alertID_about);
	    break;
	}
	return true;
	break;
    }

    return false;
}

static Boolean
PrefHandleEvent(EventPtr event)
{
    FormPtr frm;
    ListPtr list;
    ControlPtr ctl;
    Word objIndex;
    static CharPtr choices[] = { "Human vs Pilot",
				 "Pilot vs Human",
				 "Human vs Human" };

    switch (event->eType) {
    case frmOpenEvent:
	frm = FrmGetActiveForm();
	list = FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, ctlID_ModeList));
	ctl = FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, ctlID_ModeTrigger));

	if (board.playerType[0] == PlayerType_Human
	    && board.playerType[1] == PlayerType_Computer) {
	    CtlSetLabel(ctl, choices[0]);
	    LstSetSelection(list, 0);
	} else if (board.playerType[0] == PlayerType_Computer
		   && board.playerType[1] == PlayerType_Human) {
	    CtlSetLabel(ctl, choices[1]);
	    LstSetSelection(list, 1);
	} else {
	    CtlSetLabel(ctl, choices[2]);
	    LstSetSelection(list, 3);
	}

	FrmDrawForm(frm);
	return true;
	
    case ctlSelectEvent:
	switch (event->data.ctlEnter.controlID) {
	case ctlID_OkayButton:
	    frm = FrmGetActiveForm();
	    list = FrmGetObjectPtr(frm, FrmGetObjectIndex(frm,ctlID_ModeList));

	    switch (LstGetSelection(list)) {
	    case 0:
		board.playerType[0] = PlayerType_Human;
		board.playerType[1] = PlayerType_Computer;
		break;
	    case 1:
		board.playerType[0] = PlayerType_Computer;
		board.playerType[1] = PlayerType_Human;
		break;
	    case 2:
		board.playerType[0] = PlayerType_Human;
		board.playerType[1] = PlayerType_Human;
		break;
	    }
	    
	    /* fall through */

	case ctlID_CancelButton:
	    FrmReturnToForm(formID_Board);
	    return true;
	}
    }

    return false;
}

static Boolean
AppHandleEvent(EventPtr event)
{
    Word formID;
    FormPtr form;

    if (event->eType == frmLoadEvent)    {
	formID = event->data.frmLoad.formID;
	form = FrmInitForm(formID);
	FrmSetActiveForm(form);
	switch (formID) {
	case formID_Board:
	    FrmSetEventHandler(form, (FormEventHandlerPtr) BoardHandleEvent);
	    return true;
	case formID_Preferences:
	    FrmSetEventHandler(form, (FormEventHandlerPtr) PrefHandleEvent);
	    return true;
	default:
	    ErrDisplay("unhandled form ID");
	    return false;
	}
    }
    return false;
}

static void
EventLoop(void)
{
    Word err;
    EventType event;

    do {
	EvtGetEvent(&event, evtWaitForever);

	if (SysHandleEvent(&event)) continue;
	if (MenuHandleEvent((void *)0, &event, &err)) continue;
	if (AppHandleEvent(&event)) continue;
	FrmDispatchEvent(&event);

	if (boardIsOnScreen && !board.gameover)
	    MakeComputerMove();
    } while (event.eType != appStopEvent);
}

static void
StartApplication(void)
{
    Word boardSize = sizeof(board);

    boardIsOnScreen = false;
    InitBoard(&board, 3, 3);
    PrefGetAppPreferences(TicTacToeCreator, 0, &board,
			  &boardSize, false);

    FrmGotoForm(formID_Board);
}

static void
StopApplication(void)
{
    FrmCloseAllForms();
    PrefSetAppPreferences(TicTacToeCreator, 0, 0x0001,
			  &board, sizeof(board), false);
}

static Err
RomVersionCompatible(DWord requiredVersion, Word launchFlags)
{
    DWord romVersion;
    Err err;

    /* See if we have at least the minimum required version of the ROM
     * or later.
     */

    FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
    if (romVersion < requiredVersion) {
	if ((launchFlags & (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) ==
	    (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) {
	    FrmAlert(alertID_RomIncompatible);

	    /* Pilot 1.0 will continuously relaunch this app unless we
	     * switch to another safe one.
	     */
	    if (romVersion < 0x02000000)
		AppLaunchWithCommand(sysFileCDefaultApp,
				     sysAppLaunchCmdNormalLaunch, NULL);
	}
	    
	return (sysErrRomIncompatible);
    }

    return (0);
}

DWord
PilotMain (Word cmd, Ptr cmdPBP, Word launchFlags)
{
    Err err;

#define version20 sysMakeROMVersion(2,0,0,sysROMStageRelease,0)

    err = RomVersionCompatible(version20, launchFlags);
    if (err)
	return err;

    if (cmd == sysAppLaunchCmdNormalLaunch) {
	StartApplication();
	EventLoop();
	StopApplication();
    }
    return 0;
}
