/*
 * sntp client.  See rfc1361.
 * Jim Rees University of Michigan March 1998
 */

#pragma pack(2)

#include <Common.h>
#include <System/SysAll.h>
#include <System/DataMgr.h>
#include <System/NetMgr.h>
#include <UI/UIAll.h>

#include <System/Unix/unix_stdio.h>
#include <System/Unix/sys_types.h>
#include <System/Unix/sys_socket.h>
#include <System/Unix/sys_time.h>
#include <System/Unix/netinet_in.h>
#include <System/Unix/arpa_inet.h>
#include <System/Unix/netdb.h>

#include "sntp.h"

#define SntpAppID    'sNTP'
#define SntpDBType   'Data'

#define PORT 123
#define PILOTOFFSET 126144000L

struct {
    unsigned char livnmode, stratum, poll, prec;
    unsigned long delay, disp, refid;
    unsigned long reftime[2], origtime[2], rcvtime[2], txtime[2];
} pkt;

static Err OpenNet(void);
static Err CloseNet(void);
static Err OpenDatabase(void);
static Boolean sntp(EventPtr event);

FieldPtr hostfield;
FieldPtr ansfield;
FieldPtr zonefield;
DmOpenRef SntpDB;
char SntpDBName[] = "SntpDB";
int netisopen;

/* I'm not sure what these are doing here. */
Err errno, h_errno;
Long AppNetTimeout;
NetHostInfoBufType AppHostInfo;
NetServInfoBufType AppServInfo;
Word AppNetRefnum;

DWord
PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
    int err;

    if (cmd == sysAppLaunchCmdNormalLaunch) {
	err = StartApplication();
	if (err)
	    return err;
	EventLoop();
	StopApplication();
	return 0;
    } else
	return sysErrParamErr;
}

StartApplication()
{
    Err err;

    err = OpenDatabase();
    if (err)
	return err;

    FrmGotoForm(formID_sntp);

    return 0;
}

static Err
OpenNet(void)
{
    Err err;
    Word ifErrs;

    if (netisopen)
	return 0;

    err = SysLibFind("Net.lib", &AppNetRefnum);
    if (err)
	return err;

    err = NetLibOpen(AppNetRefnum, &ifErrs);
    if ((err && err != netErrAlreadyOpen) || ifErrs)
	return err;

    AppNetTimeout = SysTicksPerSecond() * 4;
    netisopen++;

    return 0;
}

static Err
CloseNet(void)
{
    if (!netisopen)
	return 0;

    NetLibClose(AppNetRefnum, false);
    netisopen = 0;
    return 0;
}

static Err
OpenDatabase(void)
{
    Err err;

    SntpDB = DmOpenDatabaseByTypeCreator(SntpDBType, SntpAppID, dmModeReadWrite);
    if (!SntpDB) {
	if ((err = DmCreateDatabase(0, SntpDBName, SntpAppID, SntpDBType, false)))
	    return err;
	SntpDB = DmOpenDatabaseByTypeCreator(SntpDBType, SntpAppID, dmModeReadWrite);
	NewRec(SntpDB, 0, 64, "");
	NewRec(SntpDB, 1, 12, "-0500");
    }
    return 0;
}

NewRec(db, i, n, s)
DmOpenRef db;
int i, n;
char *s;
{
    UInt index = i;
    VoidHand RecHandle;
    Ptr RecPointer;

    RecHandle = DmNewRecord(SntpDB, &index, n);
    RecPointer = MemHandleLock(RecHandle);
    DmWrite(RecPointer, 0, s, strlen(s) + 1);
    MemPtrUnlock(RecPointer);
    DmReleaseRecord(SntpDB, index, true);
}

CloseDatabase()
{
    FldSetTextHandle(hostfield, NULL);
    FldSetTextHandle(zonefield, NULL);
    DmReleaseRecord(SntpDB, 0, false);
    DmReleaseRecord(SntpDB, 1, false);
    DmCloseDatabase(SntpDB);
    return 0;
}

StopApplication()
{
    CloseNet();
    CloseDatabase();
}

EventLoop()
{
    short err;
    int formID;
    FormPtr form;
    EventType event;

    do {
	EvtGetEvent(&event, 200);

	if (SysHandleEvent(&event))
	    continue;
	if (MenuHandleEvent((void *)0, &event, &err))
	    continue;

	if (event.eType == frmLoadEvent) {
	    formID = event.data.frmLoad.formID;
	    form = FrmInitForm(formID);
	    FrmSetActiveForm(form);
	    switch (formID) {
	    case formID_sntp:
		FrmSetEventHandler(form, (FormEventHandlerPtr) sntp);
		break;
	    }
	}
	FrmDispatchEvent(&event);
    } while (event.eType != appStopEvent);
}

static Boolean
sntp(EventPtr event)
{
    FormPtr form = FrmGetActiveForm();
    Word field;
    int handled = 0;

    switch (event->eType) {
    case frmOpenEvent:
	hostfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_host));
	FldSetTextHandle(hostfield, (Handle) DmGetRecord(SntpDB, 0));
	zonefield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_zone));
	FldSetTextHandle(zonefield, (Handle) DmGetRecord(SntpDB, 1));
	ansfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_ans));
	FrmDrawForm(form);
	handled = 1;
	break;

    case ctlSelectEvent:  // A control button was pressed and released.
	if (event->data.ctlEnter.controlID == buttonID_query) {
	    sync(0);
	    handled = 1;
	} else if (event->data.ctlEnter.controlID == buttonID_set) {
	    sync(1);
	    handled = 1;
	}
	break;

    case menuEvent:
	field = FrmGetFocus(form);
	switch (event->data.menu.itemID) {
	case menuitemID_about:
	    FrmAlert(alertID_about);
	    break;
	case menuitemID_copy:
	    FldCopy(FrmGetObjectPtr(form, field));
	    break;
	case menuitemID_paste:
	    FldPaste(FrmGetObjectPtr(form, field));
	    break;
	}
	handled = 1;
	break;

    }
    return handled;
}

sync(int setit)
{
    int i, fd = -1, len, n;
    char hostname[64], zonename[12];
    int zonehr, zonemin, zonesign = 1;
    static char answer[32];
    struct sockaddr_in saddr, myaddr;
    struct hostent *hp;
    unsigned long ntpnow;
    long offset;

    answer[0] = '\0';
    prfield(answer, ansfield);

    /* Get server name */

    n = FldGetTextLength(hostfield);
    if (n <= 0) {
	strcpy(answer, "Please enter ntp server name");
	goto done;
    }
    bcopy(FldGetTextPtr(hostfield), hostname, n);
    hostname[n] = '\0';

    /* Get local timezone */

    n = FldGetTextLength(zonefield);
    bcopy(FldGetTextPtr(zonefield), zonename, n);
    zonename[n] = '\0';
    zonehr = atoi(zonename);
    if (zonehr < 0) {
	zonehr = -zonehr;
	zonesign = -1;
    }
    zonemin = zonehr % 100;
    zonehr = zonehr / 100;
    zonemin = zonesign * (zonehr * 60 + zonemin);

    /* Resolve server name */

    if (OpenNet()) {
	strcpy(answer, "network error");
	goto done;
    }

    bzero(&saddr, sizeof saddr);
    saddr.sin_family = AF_INET;
    saddr.sin_port = htons(PORT);
    if ((saddr.sin_addr.s_addr = inet_addr(hostname)) == -1) {
	strcpy(answer, "resolving server name...");
	prfield(answer, ansfield);
	if ((hp = gethostbyname(hostname)) == NULL) {
	    strcpy(answer, "resolve error");
	    goto done;
	}
	bcopy(hp->h_addr, &saddr.sin_addr, sizeof saddr.sin_addr);
    }

    /* Get a socket */

    if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	strcpy(answer, "socket error");
	goto done;
    }

    /* Bind it to some port */

    bzero(&myaddr, sizeof myaddr);
    myaddr.sin_family = AF_INET;
    myaddr.sin_port = htons(3030);
    myaddr.sin_addr.s_addr = INADDR_ANY;

    if (bind(fd, (struct sockaddr *) &myaddr, sizeof myaddr) < 0) {
	strcpy(answer, "bind error");
	goto done;
    }

    /* Fill in the request packet */

    bzero(&pkt, sizeof pkt);
    pkt.livnmode = 0x0b;	/* version 1, mode 3 (client) */

    /* Send packet to server */

    if ((n = sendto(fd, &pkt, sizeof pkt, 0, (struct sockaddr *) &saddr, sizeof saddr)) != sizeof pkt) {
	sprintf(answer, "sendto %d err %d", n, errno);
	goto done;
    }
    strcpy(answer, "waiting for reply...");
    prfield(answer, ansfield);

    /* Read reply packet from server */

    len = sizeof pkt;
    if ((n = recvfrom(fd, &pkt, sizeof pkt, 0, (struct sockaddr *) &saddr, &len)) != sizeof pkt) {
	if (errno == netErrTimeout)
	    strcpy(answer, "no reply from server");
	else
	    sprintf(answer, "recvfrom %d err %d", n, errno);
	goto done;
    }

    /* Sanity checks */

    if ((pkt.livnmode & 0xc0) == 0xc0 || !pkt.stratum || !pkt.txtime[0]) {
	strcpy(answer, "server not synchronized");
	goto done;
    }

    /*
     * NTP timestamps are represented as a 64-bit unsigned fixed-
     * point number, in seconds relative to 0h on 1 January 1900. The integer
     * part is in the first 32 bits and the fraction part in the last 32 bits.
     */

    ntpnow = ntohl(pkt.txtime[0]) - PILOTOFFSET + (long) zonemin * 60L;
    offset = ntpnow - TimGetSeconds();

    if (offset && setit) {
	strcpy(answer, "setting clock...");
	prfield(answer, ansfield);
	TimSetSeconds(ntpnow);
    }

    sprintf(answer, "stratum %d offset %ld", (int) pkt.stratum, offset);

 done:
    if (fd >= 0)
	close(fd);
    prfield(answer, ansfield);
}

prfield(char *s, FieldPtr field)
{
    FldSetTextPtr(field, s);
    FldRecalculateField(field, true);
    FrmDrawForm(FrmGetActiveForm());
}

/*
copyright 1998
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/
