/*
  mulg.c - a small action game for the palm pilot

  (c) 1998 Till Harbaum

  This game/code is anti-shareware.

  This means, that it is free, except for shareware authors, who have to pay
  $3 (DM 5,-) to me to register for this game and $3 to Wes Cherry (see readme.wn2)
  to register for his win2 library. Any freeware author is allowed to include 
  code fragments into his programs (see readme.wn2 for the win2 part).

  Version 1.0  - initial release
          1.1  - fixed fatal exception when entering border area (happened in level 5)
	  1.1b - fixed error with document in level 15
	  1.1c - works with game sound preferences

  ToDo for future versions:
  - exclude diagonal leaving of tiles from collision test
  - more levels
  - database support
  - sprite redraw completely different 
    (without background buffer but the ability to
    work with multi layer tiles und 'tunnel' tiles)
  - Conways game of life (birth = empty+3, stay = 2/3, death = >3)
*/

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>

#include "win2.h"

#include "rsc.h"
#include "mulg.h"
#include "tiles.h"

#define PREFS_VERSION 2   /* version of preferences structure */

#define SENSE 5           /* pen sensivity */
#define SLOW  95          /* every step reduces marble energy by 5% (due to friction) */
#define DUSCH 70          /* with every hit to a wall, the marble looses 30% energie */
#define HOLED 3           /* acceleration for hole/hump */
#define WIPS  0           /* movement of the seesaw (0=toggles exactly in the middle) */
#define WIPD 0x20         /* acceleration due to seesaw */
#define VENT_ACC 20       /* acceleration due to running ventilator */
#define PUSH  1000        /* force needed to push a box */
#define BUMPACC 1000      /* acceleration due to bumper */
#define BUMPACCD 700      /* diagonal acceleration due to bumper (1/SQR(2)*BUMPACC) */
#define BOMB_TIMEOUT  20
#define BOMB_TIMEOUT2 5

/* constants for acceleration table for holes and humps (hole.h) */
/* and collision table */
#define DIR_OL 0x80
#define DIR_UL 0x40
#define DIR_UR 0x20 
#define DIR_OR 0x10

/* game states */
#define SETUP   0
#define RUNNING 1
#define MESSAGE 2

/* movement extras */
#define GND_ICE     1  /* whoa, this is slippery */
#define GND_REVERSE 2  /* and this is stupid */
#define CARRY_MATCH 4  /* carrying ignited match */

#define MAX_COLLECT 5

/* tile flags */
#define ID_MASK   0x1f  /* allowing up to 32 switch/object pairs */

#define PSTART    0xff  /* PATH */
#define REVERSE   0x80  /* PATH/BOXFIX/ICE */
#define UNREVERSE 0x40  /* PATH/BOXFIX/ICE */
#define OPENING   0x80  /* DOOR */
#define CLOSING   0x40  /* DOOR */
#define ROTATING  0x80  /* VENTILATOR */

/* load levels from internal data */
extern unsigned short *level[];
extern int level_size[][2];
extern char *level_name[];
extern char *message[];

extern unsigned char hole[][16];

/* game functions */
void draw_tile(int tile, int x, int y);
void draw_level(int xp, int yp);
void create_sprite(int no, int tile, int hs_x, int hs_y);
void init_sprites(void);
void undraw_sprite(int no);
void redraw_sprite(int no, int x, int y);
void draw_sprite(int no, int x, int y);
int  disp_grey(void);

FormPtr pfrmMain;

unsigned int   marble_x,  marble_y, marble_xp, marble_yp;
unsigned char  level_buffer[MAX_HEIGHT][MAX_WIDTH];      /* max 37 x 33 level -> 1221 bytes */
unsigned char  attribute_buffer[MAX_HEIGHT][MAX_WIDTH];  /* max 37 x 33 level -> 1221 bytes */
unsigned short level_width, level_height;

unsigned long level_time;
int game_state=SETUP;
int collected[MAX_COLLECT];

const int x_add[]={ 0,10,19,28};
const int y_add[]={ 0, 9,17,25};
const unsigned char hole_acc[]={0x0,0x1,0x3,0x7,0xf};

/* to be saved in preferences: */
struct {
  int version, level_no, max_enabled;
  unsigned long hiscore[LEVELS];
} prefs;

int marble_sx, marble_sy;
int pen_last_x, pen_last_y;
int marble_ay, marble_ax, marble_extra;

int sound_on;

struct SndCommandType snd_switch={sndCmdFreqDurationAmp,100,1,sndMaxAmp};

/* game time to ascii string */
void time2str(long time, char *str) {
  str[0]='0'+(time/600000)%10;
  str[1]='0'+(time/60000)%10; 
  str[3]='0'+(time/10000)%6;
  str[4]='0'+(time/1000)%10;
  str[6]='0'+(time/100)%10;
}

/* start a new game */
void init_level(void) {
  int y,x;
  unsigned short *src;

  level_width  = level_size[prefs.level_no][0];
  level_height = level_size[prefs.level_no][1];

  /* clear level buffer */
  for(y=0;y<MAX_HEIGHT;y++)
    for(x=0;x<MAX_WIDTH;x++)
      level_buffer[y][x]=attribute_buffer[y][x]=ZERO;

  /* copy all level information */
  src = level[prefs.level_no];

  for(y=0;y<level_height;y++) {
    for(x=0;x<level_width;x++) {
      level_buffer[y][x]     = (unsigned char)*src;
      attribute_buffer[y][x] = (unsigned char)((*src++)>>8);
    }
  }
  
  /* init collect buffer */
  for(x=0;x<MAX_COLLECT;x++)
    collected[x]=ZERO;

  /* start level time counter */
  level_time=0;
}

/* add an object to the collection buffer */
int add_to_collection(int type) {
  int i,ret=false;

  for(i=0;(i<MAX_COLLECT)&&(!ret);i++) {
    if(collected[i]==ZERO) {
      collected[i]=type;
      ret=true;
      if(type==MATCHX)
	marble_extra|=CARRY_MATCH;
    }
  }
  return ret;
}

/* returns the x-th object from the collection buffer */
/* and removes it if it is a doc */
int get_from_collection(int x) {
  int i,ret=ZERO;

  if((collected[x]&0xff)==DOCX) {
    ret=collected[x];
    for(i=x;i<(MAX_COLLECT-1);i++)
      collected[i]=collected[i+1];

    collected[MAX_COLLECT-1]=ZERO;
  }
  return(ret);
}

/* find the object obj in the collection buffer and remove it */
int get_object_from_collection(int obj) {
  int i,j;

  for(i=0;i<MAX_COLLECT-1;i++) {

    /* found??? */
    if((collected[i]&0xff)==obj) {

      /* remove object */
      for(j=i;j<(MAX_COLLECT-1);j++)
	collected[j]=collected[j+1];

      collected[MAX_COLLECT-1]=ZERO;      
      return true;
    }
  }
  return(false);
}

void draw_collection(void) {
  int i;
  for(i=0;i<MAX_COLLECT;i++)
    draw_tile(collected[i]&0xff, i, 9);
}

int get_collection(int x, int y) {
  return ZERO;
}

void init_marble(void) {
  int x,y;
  
  for(y=0;y<level_height;y++)
    for(x=0;x<level_width;x++)
      /* start tile? */
      if((level_buffer[y][x] == PATH)&&
	 (attribute_buffer[y][x] == PSTART)) {
	marble_xp = ((x-1)/9)*9;                /* horizontal page offset */
	marble_x  = ((16*(x-marble_xp)+7)<<8);  /* horizontal position */
	marble_yp = ((y-1)/8)*8;                /* vertical page offset */
	marble_y  = ((16*(y-marble_yp)+7)<<8);  /* vertical position */
	attribute_buffer[y][x]=0;               /* clear start flag */
      }
  
  /* start acceleration */
  marble_ax=marble_sx=0;
  marble_ay=marble_sy=0;
  marble_extra = 0;
}

/* restart everything */
void init_game(void) {
  /* start with level 0 */
  init_level();
  init_marble();
  init_sprites();
  create_sprite(MARBLE,BALL,7,7);
}

char time_msg[]="Time: 00:00.0";
void draw_state(void) {
  int i;

  if((game_state==RUNNING)||(game_state==MESSAGE)) {
    draw_collection();
    time2str(level_time ,&time_msg[6]);
    Win2DrawChars(time_msg, StrLen(time_msg), 100, 146);
  }
}

struct SndCommandType snd_kill={sndCmdFreqDurationAmp,100,1,sndMaxAmp};
char no_score[]="not yet played";

void fill_form(void) {
  char no_str[4];
  RectangleType rect;

  /* level no to string */
  FntSetFont(2);
  no_str[0] = '0' + (prefs.level_no+1)/10;
  no_str[1] = '0' + (prefs.level_no+1)%10;
  no_str[2] = ':';
  no_str[3] = 0;

  /* output level no and name */
  rect.topLeft.x=10; rect.topLeft.y=81; 
  rect.extent.x=127; rect.extent.y=16;
  WinEraseRectangle(&rect,0);
  WinDrawChars(no_str, 3, 10, 81);
  WinDrawChars(level_name[prefs.level_no], StrLen(level_name[prefs.level_no]), 30, 81);

  /* output time */
  rect.topLeft.y=117;
  WinEraseRectangle(&rect,0);

  if(prefs.hiscore[prefs.level_no]!=0xffffffff) {
    time2str(prefs.hiscore[prefs.level_no],&time_msg[6]);
    WinDrawChars(&time_msg[6], StrLen(&time_msg[6]), 10, 117);
  } else
    WinDrawChars(no_score, StrLen(no_score), 10, 117);

  FntSetFont(0);
}

void game_end(int killed) {
  int i;

  if(game_state==RUNNING) {

    if(killed) {
      undraw_sprite(MARBLE);

      if(sound_on) {
	/* do sound effect */
	for(i=400;i>100;i-=6) {
	  snd_kill.param1=i;
	  SndDoCmd(NULL, &snd_kill, 0); 
	}
      }
    }
    /* wait one second */
    SysTaskDelay(SysTicksPerSecond());

    game_state=SETUP;
    Win2SetMono();

    if(!killed) {
      /* new high score?? */
      if(level_time < prefs.hiscore[prefs.level_no]) {
	FrmAlert(HiScore);
	prefs.hiscore[prefs.level_no]=level_time;
      }

      /* never played the next level?? */
      if(prefs.max_enabled==prefs.level_no) {
	prefs.level_no++;
	/* enable next level */
	prefs.max_enabled=prefs.level_no;

	/* limit level no */
	if(prefs.level_no>(LEVELS-1))
	  prefs.level_no=(LEVELS-1);
      }
      /* refill form */
      fill_form();
    }
  }
}

void snd_clic(void) {
  if(sound_on)
    SndDoCmd(NULL,&snd_switch,0);
}

#define MSG_WIDTH  90
#define MSG_HEIGHT 100
#define MSG_BORDER 5
void draw_message(int no) {
  int lines,wr,y;
  RectangleType rc;

  game_state=MESSAGE;
  snd_clic();

  /* draw white rectangle */
  Win2SetColor(clrLtGrey);
  rc.topLeft.x=(160-MSG_WIDTH)/2;  rc.extent.x=MSG_WIDTH; rc.topLeft.y=(160-MSG_HEIGHT)/2; rc.extent.y=MSG_HEIGHT;
  Win2FillRect(&rc,0,0);

  /* draw border */
  Win2SetColor(clrWhite);
  rc.extent.x=MSG_WIDTH; rc.extent.y=1;
  Win2FillRect(&rc,0,3);
  rc.extent.x=1;         rc.extent.y=MSG_HEIGHT;
  Win2FillRect(&rc,0,3);

  Win2SetColor(clrBlack);
  rc.extent.x=MSG_WIDTH; rc.topLeft.y=(160+MSG_HEIGHT)/2; rc.extent.y=1;
  Win2FillRect(&rc,0,3);
  rc.topLeft.x=(160+MSG_WIDTH)/2;  rc.extent.x=1; rc.topLeft.y=(160-MSG_HEIGHT)/2; rc.extent.y=MSG_HEIGHT+1;
  Win2FillRect(&rc,0,3); 

  Win2SetColor(clrBlack);
  /* determine no of lines */
  lines=0; wr=0;
  while(wr<StrLen(message[no])) {
    wr+=FntWordWrap(&message[no][wr], MSG_WIDTH-2*MSG_BORDER);
    lines++;
  }
  
  /* draw lines */
  y=80-(FntLineHeight()*(lines/2)); wr=0;
  while(wr<StrLen(message[no])) {
    lines=FntWordWrap(&message[no][wr], MSG_WIDTH-2*MSG_BORDER);
    Win2DrawChars(&message[no][wr],lines, 80-(FntLineWidth(&message[no][wr],lines)/2), y);
    wr+=lines;
    y+=FntLineHeight();
  }
}

/* check if tile at x,y is visible on screen */
int tile_is_visible(unsigned int x, unsigned int y) {
  if(((x-marble_xp)<10)&&
     ((y-marble_yp)<9))
    return 1;
  
  return 0;
}

/* draw tile only if it is currently visible */
void draw_tile_if_visible(unsigned short tile, int x, int y) { 
  if(tile_is_visible(x,y))
    draw_tile(tile, x-marble_xp, y-marble_yp);
}

/* check if there is a running ventilator */
int check_ventilator(unsigned int x, unsigned int y) {
  unsigned char st;

  /* since this is unsigned, values <0 are >MAX, too */
  if((x>=MAX_WIDTH)||(y>=MAX_HEIGHT))
    return 0;

  st=level_buffer[y][x];

  /* this is a nice place for explosion check, too */
  if(st==EXPLODE)
    game_end(true);
  
  /* is there a ventilator? */
  if((st>=HVENT_1)&&(st<=HVENT_4)) {

    /* is it running? */
    if(attribute_buffer[y][x]&ROTATING) {
      
      /* remove all matches from collection */
      if(marble_extra&CARRY_MATCH) {
	marble_extra&=~CARRY_MATCH;
	while(get_object_from_collection(MATCHX));
      }

      return 1;
    }
  }
  return 0;
}

void switch_on(int id) {
  int   ix,iy;
  
  snd_clic();

  /* search for associated object */
  for(iy=0;iy<level_height;iy++) {
    for(ix=0;ix<level_width;ix++) {
      /* object found */
      if((attribute_buffer[iy][ix]&ID_MASK)==id) {
	switch(level_buffer[iy][ix]) {
		
	  /* shut down ventilator */
	case HVENT_1:
	case HVENT_2:
	case HVENT_3:
	case HVENT_4:
	  attribute_buffer[iy][ix]&=~ROTATING;
	  break;
	  
	  /* open door */
	case HDOOR_1:
	case VDOOR_1:
	case HDOOR_2:
	case VDOOR_2:
	case HDOOR_3:
	case VDOOR_3:
	  attribute_buffer[iy][ix]=(attribute_buffer[iy][ix]&~(OPENING|CLOSING))|OPENING;
	  break;

	  /* close hole */
	  /* open inversed hole */
	case PATH:
	case SPACE:
	  level_buffer[iy][ix]=(level_buffer[iy][ix]==PATH)?SPACE:PATH;
	  draw_tile_if_visible(level_buffer[iy][ix], ix, iy);
	  break;
	}
      }
    }
  }
}

void switch_off(int id) {
  int   ix,iy;

  snd_clic();

  /* search for associated object */
  for(iy=0;iy<level_height;iy++) {
    for(ix=0;ix<level_width;ix++) {
      /* object found */
      if((attribute_buffer[iy][ix]&ID_MASK)==id) {
	switch(level_buffer[iy][ix]) {
	  
	  /* ventilator on */
	case HVENT_1:
	case HVENT_2:
	case HVENT_3:
	case HVENT_4:
	  attribute_buffer[iy][ix]|=ROTATING;
	  break;
	      
	  /* close door */
	case HDOOR_2:
	case VDOOR_2:
	case HDOOR_3:
	case VDOOR_3:
	case HDOOR_4:
	case VDOOR_4:
	  attribute_buffer[iy][ix]=(attribute_buffer[iy][ix]&~(OPENING|CLOSING))|CLOSING;
	  break;
	      
	  /* open hole */
	  /* close inversed hole */
	case PATH:
	case SPACE:
	  level_buffer[iy][ix]=(level_buffer[iy][ix]==PATH)?SPACE:PATH;
	  draw_tile_if_visible(level_buffer[iy][ix], ix, iy);
	  break;
	}
      }
    }
  }
}

int check_tile(unsigned int x, unsigned int y, int dir) {
  short id;
  unsigned char st;
  int lx,ly, t,l;
  
  /* since this is unsigned, values <0 are >MAX, too */
  if((x>=MAX_WIDTH)||(y>=MAX_HEIGHT))
    return 0;

  st=level_buffer[y][x];

  switch(st) {
    /* stone -> no way */
  case STONE: 
    /* horizontal door (not open) */
  case HDOOR_1:
  case HDOOR_2:
  case HDOOR_3:

    /* vertical door */
  case VDOOR_1:
  case VDOOR_2:
  case VDOOR_3:
    return 1;

  case HVENT_1:
  case HVENT_2:
  case HVENT_3:
  case HVENT_4:
    /* killed by running fan */
    if(attribute_buffer[y][x]&ROTATING) game_end(true);
    return 0;

    /* one ways */
  case OWR:  /* one-way right */
    return(!((dir==1)||(dir==0)));    

  case OWL:  /* one-way left */
    return(!((dir==5)||(dir==0)));    

  case OWU:  /* one-way up */
    return(!((dir==3)||(dir==0)));    

  case OWD:  /* one-way down */
    return(!((dir==7)||(dir==0)));    

  case KEYHOLE:
    snd_clic();

    /* change keyhole to switch */
    if(get_object_from_collection(KEYX)) {
      level_buffer[y][x] = SWITCH_ON;
      draw_tile_if_visible(level_buffer[y][x], x, y);    
    }
    return 1;  
    
  case SWITCH_ON:
    switch_on((attribute_buffer[y][x])&ID_MASK);

    /* switch is now off */
    level_buffer[y][x] = SWITCH_OFF;
    draw_tile_if_visible(level_buffer[y][x], x, y);
    return 1;  
    
  case SWITCH_OFF:
    switch_off((attribute_buffer[y][x])&ID_MASK);

    /* switch is now on */
    level_buffer[y][x] = SWITCH_ON;
    draw_tile_if_visible(level_buffer[y][x], x, y);
    return 1;  
	
    case DICE1:
    case DICE2:
    case DICE3:
    case DICE4:
    case DICE5:
    case DICE6:
      /* randomize the dice */
      level_buffer[y][x] = DICE1 + SysRandom(0)%6;
      draw_tile_if_visible(level_buffer[y][x], x, y);
      
      /* look if all dices are equal */
      l=1;
      for(ly=0;ly<level_height;ly++) {
	for(lx=0;lx<level_width;lx++) {
	  t = level_buffer[ly][lx];
	  /* is this a dice? */
	  if((t>=DICE1)&&(t<=DICE6)) {
	    /* same object identifier */
	    if(attribute_buffer[ly][lx] == attribute_buffer[y][x]) {
	      /* same value?? */
	      if(t!=level_buffer[y][x]) l=0;  /* nope! */
	    }
	  }
	}
      }
      /* yupp, all dices are equal */
      if(l) switch_on (attribute_buffer[y][x]&ID_MASK);
      else  switch_off(attribute_buffer[y][x]&ID_MASK);
      return 1;

  case SKULL:
    game_end(true);
    return 1;

  case BUMP:
  case BUMPL:
    snd_clic();

    if(dir==1)   marble_ax=-BUMPACC; 
    if(dir==2) { marble_ax=-BUMPACCD; marble_ay= BUMPACCD; };
    if(dir==3)                        marble_ay= BUMPACC; 
    if(dir==4) { marble_ax= BUMPACCD; marble_ay= BUMPACCD; };
    if(dir==5)   marble_ax= BUMPACC; 
    if(dir==6) { marble_ax= BUMPACCD; marble_ay=-BUMPACCD; };
    if(dir==7)                        marble_ay=-BUMPACC; 
    if(dir==8) { marble_ax=-BUMPACCD; marble_ay=-BUMPACCD; };

    /* activate bumper */
    if(marble_ax+marble_ay!=0) {
      level_buffer[y][x]=BUMPL;
      draw_tile_if_visible(level_buffer[y][x], x, y);
    }
    return 1;

  case BOX:
    snd_clic();

    /* push box to the next field if possible (speed>PUSH) */ 
    lx=x; ly=y;
    if((dir==1)&&(marble_sx> PUSH)) lx++;    /* from the left */
    if((dir==3)&&(marble_sy<-PUSH)) ly--;    /* from the bottom */
    if((dir==5)&&(marble_sx<-PUSH)) lx--;    /* from the right */
    if((dir==7)&&(marble_sy> PUSH)) ly++;    /* from the top */

    if((x!=lx)||(y!=ly)) {
      /* is this a tile, a box can be pushed onto? */
      t=level_buffer[ly][lx];
      if((t==KEY)||(t==PATH)||(t==HUMP)||(t==HOLE)||(t==SPACE)||
	 (t==BOXFIX)||(t==ICE)||((t>=OWL)&&(t<=OWD))) {

	if(t==SPACE)
	  level_buffer[ly][lx]=BOXFIX;  /* fill space with box */
	else {
	  /* save whats under the box in attribute */
	  /* (therefore no DOC possible here :-( ) */
	  attribute_buffer[ly][lx] = level_buffer[ly][lx];
	  level_buffer[ly][lx] = BOX;
	}

	/* remove old box */
	level_buffer[y][x] = attribute_buffer[y][x];
	attribute_buffer[y][x] = 0;

	/* and redraw both */
	draw_tile_if_visible(level_buffer[ly][lx], lx, ly);
	draw_tile_if_visible(level_buffer[y][x], x, y);
       }
    }
    return 1;

  case WIPPR:
  case WIPPL:
  case WIPPU:
  case WIPPO:
    /* seesaw -> determine direction of entering */
    if(dir==0) return 0;
    if((st==WIPPL)&&(dir==1))  return 0;  /* from the left onto WIPPR is ok */
    if((st==WIPPU)&&(dir==3))  return 0;  /* from the bottom onto WIPPO is ok */
    if((st==WIPPR)&&(dir==5))  return 0;  /* from the right onto WIPPL is ok */
    if((st==WIPPO)&&(dir==7))  return 0;  /* from the top onto WIPPU is ok */
    
    return 1;
  }
  
  return 0;
}


int check_marble(void) {
  /* coordinates of current tile */
  int ix, iy, x, y, t;
  short st;

  /* leaving the game area? need to switch to new game area? */
  if((marble_x>>8)==152) {      /* at right */
    marble_xp+=9;
    marble_x-=(144<<8);         /* beam marble to the very left */
    draw_level(marble_xp, marble_yp);
  }

  if((marble_x>>8)==6) {        /* at left */
    marble_xp-=9;
    marble_x+=(144<<8);         /* beam marble to the very right */
    draw_level(marble_xp, marble_yp);
  }

  if((marble_y>>8)==6) {        /* at top */
    marble_yp-=8;
    marble_y+=(128<<8);         /* beam marble to the very bottom */
    draw_level(marble_xp, marble_yp);
  }

  if((marble_y>>8)==136) {      /* at bottom */
    marble_yp+=8;
    marble_y-=(128<<8);         /* beam marble to the very top */
    draw_level(marble_xp, marble_yp);
  }

  ix = (marble_x>>8)&15; x = (marble_x>>8)/16 + marble_xp;
  iy = (marble_y>>8)&15; y = (marble_y>>8)/16 + marble_yp;

  st=level_buffer[y][x];

  if((st==DOC)||(st==KEY)||(st==MATCH)) {
    /* collectables (document, key, match) */
    if(add_to_collection(( (int)(attribute_buffer[y][x])<<8)|(st+1))) {
      snd_clic();

      draw_collection();

      /* remove tile */
      level_buffer[y][x]=PATH;
      draw_tile_if_visible(PATH, x, y);
    }
  }

  /* igniting unignited bomb */
  if((st==BOMB)&&(attribute_buffer[y][x]==0)&&(marble_extra&CARRY_MATCH))
    attribute_buffer[y][x]=BOMB_TIMEOUT;

  /* walking on ice */
  if(st==ICE)
    marble_extra|=GND_ICE;

  /* reversed PATH/BOXFIX/... */
  if((attribute_buffer[y][x]&REVERSE)||(attribute_buffer[y][x]&UNREVERSE)) {
    if((st==BOXFIX)||(st==PATH)||(st==ICE)) {
      if(attribute_buffer[y][x]&REVERSE)
	marble_extra|=GND_REVERSE;
      else
	marble_extra&=~GND_REVERSE;
    }
  }

  /* lost in space -> end of game */
  if(st==SPACE)
    game_end(true);

  /* reached the center of the goal */
  if((st==GOAL)&&(ix>=5)&&(ix<=9)&&(iy>=5)&&(iy<=9))
    game_end(false);

  /* the seesaw -> may toggle */
  if((st==WIPPR)||(st==WIPPL)||(st==WIPPU)||(st==WIPPO)) {
    t=0;
    if(st==WIPPR) {  /* right seesaw */
      marble_sx += WIPD;
      if(ix<(7-WIPS)) t=WIPPL;
    }
    if(st==WIPPL) {  /* left seesaw */
      marble_sx -= WIPD;
      if(ix>(7+WIPS)) t=WIPPR;
    }
    if(st==WIPPU) {  /* top seesaw */
      marble_sy += WIPD;
      if(iy<(7-WIPS)) t=WIPPO;
    }
    if(st==WIPPO) {  /* bottom seesaw */
      marble_sy -= WIPD;
      if(iy>(7+WIPS)) t=WIPPU;
    }

    /* seesaw toggled? redraw! */
    if(t) {
      level_buffer[y][x] = t;
      draw_tile_if_visible(level_buffer[y][x], x, y);
    }
  }

  /* is the marble on a hole or hump?? */
  if((st==HOLE)||(st==HUMP)) {
    t=((st==HUMP)?-1:1);
    marble_sx += t*((ix>7)?-1:1)*((hole_acc[hole[ix][iy]&0x07])<<HOLED);
    marble_sy += t*((iy>7)?-1:1)*((hole_acc[hole[iy][ix]&0x07])<<HOLED);
  }

  /* see if there is a ventilator (or explosion) near */
  if(check_ventilator(x+1, y  )) marble_sx+=VENT_ACC;
  if(check_ventilator(x  , y+1)) marble_sy+=VENT_ACC;
  if(check_ventilator(x-1, y  )) marble_sx-=VENT_ACC;
  if(check_ventilator(x  , y-1)) marble_sy-=VENT_ACC;

  /* direction value:                                */
  /*    8 7 6   these values are set when _entering_ */
  /*     \|/    a tile, not when already standing on */
  /*    1-X-5   that tile                            */
  /*     /|\                                         */
  /*    2 3 4                                        */

  /* see, if there is something blocking our way ... */

  /* ... horizontally ... */
  if(((ix>9)&&(check_tile(x+1,y,((ix==10)&&(marble_sx>0))?1:0)))||
     ((ix<5)&&(check_tile(x-1,y,((ix== 4)&&(marble_sx<0))?5:0)))) {
    marble_sx = (DUSCH*(long)marble_sx)/100;
    return 1;
  }

  /* ... vertically ... */
  if(((iy>9)&&(check_tile(x,y+1,((iy==10)&&(marble_sy>0))?7:0)))||
     ((iy<5)&&(check_tile(x,y-1,((iy== 4)&&(marble_sy<0))?3:0)))) {
    marble_sy = (DUSCH*(long)marble_sy)/100;
    return 1;
  }

  /* ... and diagonally ... */
  if((t=hole[iy][ix])&0xf8) {
    if(((t&DIR_OL)&&(check_tile(x-1,y-1,(t&8)?4:0)))||
       ((t&DIR_UL)&&(check_tile(x-1,y+1,(t&8)?6:0)))||
       ((t&DIR_OR)&&(check_tile(x+1,y-1,(t&8)?2:0)))||
       ((t&DIR_UR)&&(check_tile(x+1,y+1,(t&8)?8:0)))) {
      marble_sx = (DUSCH*(long)marble_sx)/100;
      marble_sy = (DUSCH*(long)marble_sy)/100;
      return 1;
    }
  }

  return 0;
}

void move_marble(Word *x, Word *y) {
  int r,step_x, step_y, ist_x, ist_y;
  int marble_ox, marble_oy;

  ist_x=0; ist_y=0;

  marble_ox=marble_sx; marble_oy=marble_sy;

  if((ABS(marble_sx>=256)||(ABS(marble_sy)>=256))) {
    if(ABS(marble_sx)>=ABS(marble_sy)) {

      /* step in x direction */
      step_y=256*(long)marble_sy/(long)ABS(marble_sx);
      step_x=(marble_sx>0)?256:-256;
      for(r=ABS(marble_sx/256);r>0;r--) {
	ist_y+=ABS(step_y); 
	marble_y+=step_y; 
	if(check_marble()) {
	  step_y=-step_y;
	  marble_sy=-marble_sy;
	  marble_y+=2*step_y;
	}
	ist_x+=ABS(step_x); 
	marble_x+=step_x;
	if(check_marble()) {
	  step_x=-step_x;
	  marble_sx=-marble_sx;
	  marble_x+=2*step_x;
	}
      }
    } else {

      /* step in Y direction */
      step_x=256*(long)marble_sx/(long)ABS(marble_sy);
      step_y=(marble_sy>0)?256:-256;
      for(r=ABS(marble_sy/256);r>0;r--) {
	marble_y+=step_y;
	ist_y+=ABS(step_y);
	if(check_marble()) {
	  step_y=-step_y;
	  marble_sy=-marble_sy;
	  marble_y+=2*step_y;
	}
	marble_x+=step_x;
	ist_x+=ABS(step_x);
	if(check_marble()) {
	  step_x=-step_x;
	  marble_sx=-marble_sx;
	  marble_x+=2*step_x;
	}
      }
    }
  }

  /* remaining step X */
  if(ist_x<ABS(marble_ox)) {
    marble_x += (ABS(marble_ox)-ist_x)*SGN(marble_sx);
    if(check_marble()) {
      marble_sx=-marble_sx;
      marble_x += 2*((ABS(marble_ox)-ist_x)*SGN(marble_sx));
    }	  
  }

  /* remaining step Y */
  if(ist_y<ABS(marble_oy)) {
    marble_y += (ABS(marble_oy)-ist_y)*SGN(marble_sy);
    if(check_marble()) {
      marble_sy=-marble_sy;
      marble_y += 2*((ABS(marble_oy)-ist_y)*SGN(marble_sy));
    }
  }

  /* walking on ice? */
  if(!(marble_extra&GND_ICE)) {
    marble_sx = (SLOW*(long)marble_sx)/100;
    marble_sy = (SLOW*(long)marble_sy)/100;
  }

  /* new acceleration applied by bumper? */
  if(marble_ax!=0) { marble_sx=marble_ax; marble_ax=0; }
  if(marble_ay!=0) { marble_sy=marble_ay; marble_ay=0; }
  marble_extra&=(GND_REVERSE|CARRY_MATCH);  /* clear all flags except REVERSE and CARRY_MATCH */

  *x=marble_x>>8;
  *y=marble_y>>8;
} 

/* user pushed marble */
void marble_push(int x, int y) {

  /* on force reversing tile? */
  if(marble_extra & GND_REVERSE) {
    marble_sx -= (x<<SENSE);
    marble_sy -= (y<<SENSE);
  } else {
    marble_sx += (x<<SENSE);
    marble_sy += (y<<SENSE);
  }
}

void ignite(int x, int y) {
  switch(level_buffer[y][x]) {

    /* explosion ignites neighbouring bombs */
  case BOMB:
    if(attribute_buffer[y][x]==0)
      attribute_buffer[y][x]=BOMB_TIMEOUT2;
    break;

    /* stone becomes free path */
  case STONE:
    level_buffer[y][x]=PATH;
    draw_tile_if_visible(level_buffer[y][x], x, y);
    break;

    /* ice becomes empty space */
  case ICE:
    level_buffer[y][x]=SPACE;
    draw_tile_if_visible(level_buffer[y][x], x, y);
    break;
  }
}

/* do the animations */
void do_animations(void) {
  int p,x,y;
  static int ani_cnt=0;

  for(y=0;y<level_height; y++) {
    for(x=0;x<level_width; x++) {
      switch(level_buffer[y][x]) {

	/* count down bomb */
      case BOMB:
      case BOMBI:
	if(!(ani_cnt&3)) {
	  if(attribute_buffer[y][x]>0) {
	    attribute_buffer[y][x]-=1;

	    if(attribute_buffer[y][x]!=0)
	      level_buffer[y][x]=(attribute_buffer[y][x]&1)?BOMB:BOMBI;
	    else
	      level_buffer[y][x]=EXPLODE;

	    draw_tile_if_visible(level_buffer[y][x], x, y);
	  }
	}
	break;

	/* explosion */
      case EXPLODE:
	if(!(ani_cnt&3)) {
	  snd_clic();
	  level_buffer[y][x]=HOLE;	
	  draw_tile_if_visible(level_buffer[y][x], x, y);

	  /* explosion ignites neighbour */
	  ignite(x+1,y);
	  ignite(x,y+1);
	  ignite(x-1,y);
	  ignite(x,y-1);
	}
	break;

	/* flash bumper */
      case BUMPL:
	level_buffer[y][x]=BUMP;
	draw_tile_if_visible(level_buffer[y][x], x, y);
	break;
    
	/* door */
      case HDOOR_1:
      case HDOOR_2:
      case HDOOR_3:
      case HDOOR_4:
      case VDOOR_1:
      case VDOOR_2:
      case VDOOR_3:
      case VDOOR_4:
	if(!(ani_cnt&3)) {
	  /* opening door */
	  if(attribute_buffer[y][x]&OPENING) {
	    if((level_buffer[y][x]!=HDOOR_4)&&
	       (level_buffer[y][x]!=VDOOR_4)) {
	      level_buffer[y][x]+=1;
	      draw_tile_if_visible(level_buffer[y][x], x, y);
	    } else
	      /* door is open */
	      attribute_buffer[y][x]&=~OPENING;
	  }
	     
	  /* closing door */
	  if(attribute_buffer[y][x]&CLOSING) {
	    if((level_buffer[y][x]!=HDOOR_1)&&
	       (level_buffer[y][x]!=VDOOR_1)) {
	      level_buffer[y][x]-=1;
	      draw_tile_if_visible(level_buffer[y][x], x, y);
	    } else
	      /* door is closed */
	      attribute_buffer[y][x]&=~CLOSING;
	  }
	}
	break;
	    
	/* rotate ventilator */
      case HVENT_1:
      case HVENT_2:
      case HVENT_3:
      case HVENT_4:
	if((!(ani_cnt&3))&&(attribute_buffer[y][x]&ROTATING)) {
	  /* counter up */
	  if(level_buffer[y][x]<HVENT_4) 
	    level_buffer[y][x]+=1;
	  else                     
	    level_buffer[y][x]=HVENT_1;
	  
	  /* draw tile */
	  draw_tile_if_visible(level_buffer[y][x], x, y);
	}
	break;
      }
    }
  }
  /* increase animation counter */
  ani_cnt++;
}

static Boolean MainFormHandleEvent(EventPtr event)
{
  Boolean handled = false;
  int i;
  
  switch (event->eType) {

  case ctlRepeatEvent:
    if(event->data.ctlEnter.controlID == LevelUp) {
      prefs.level_no++;
      if(prefs.level_no>prefs.max_enabled) prefs.level_no=prefs.max_enabled;
      if(prefs.level_no>(LEVELS-1))        prefs.level_no=LEVELS-1;
      
      fill_form();
    }
    
    if(event->data.ctlEnter.controlID == LevelDown) {
      if(--prefs.level_no<0) 
	prefs.level_no=0;
      else
	fill_form();
    }
    break;
 
  case ctlSelectEvent:
    if(event->data.ctlEnter.controlID == PlayButton) {

      game_state=RUNNING;
      init_game();

      if (Win2SetGreyscale() != 0) {
	FrmAlert(RscOom);
	game_state=SETUP;
      } else {
	draw_level(marble_xp, marble_yp);
	draw_sprite(MARBLE, marble_x>>8, marble_y>>8);
	draw_state();
	      
	if(sound_on) {
	  /* start sound */
	  for(i=100;i<400;i+=6) {
	    snd_kill.param1=i;
	    SndDoCmd(NULL, &snd_kill, 0); 
	  }
	}
      }
      handled=true;
    }
    break;
    
  }
  return handled;
}

DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags) {
  short err;
  EventType e;
  FormType *pfrm;
  Boolean fMainWindowActive;
  Word  ticks_wait,x,y;
  long  last_ticks=0;
  Word  size;
  SystemPreferencesType sysPrefs;

  if (!cmd) {
    for(x=0;x<LEVELS;x++) prefs.hiscore[x]=0xffffffff;      

    size = sizeof(prefs);
    x=PrefGetAppPreferences( CREATOR, 0, &prefs, &size, true);

    if((x==noPreferenceFound)||(prefs.version!=PREFS_VERSION)) {
      /* level selection starts with level 0 */
      prefs.version = PREFS_VERSION;
      prefs.level_no = 0;
      prefs.max_enabled = 0;
      for(x=0;x<LEVELS;x++) prefs.hiscore[x]=0xffffffff;      
    }

    /* get system preferences for sound */
    PrefGetPreferences(&sysPrefs);
    sound_on=(sysPrefs.gameSoundLevelV20 == slOn)?1:0;

    /* initialize Timer */
    ticks_wait = SysTicksPerSecond()/50;  /* 20ms/frame */

    FrmGotoForm(RscForm1);

    while(1) {
      EvtGetEvent(&e, ticks_wait);

      if(game_state!=SETUP)
	if (Win2PreFilterEvent(&e))
	  continue;

      if (SysHandleEvent(&e)) 
	continue;

      if (MenuHandleEvent((void *)0, &e, &err)) 
	continue;
	
      switch (e.eType) {

      case frmLoadEvent:	
	pfrm = FrmInitForm(e.data.frmLoad.formID);
	if (e.data.frmLoad.formID == RscForm1) pfrmMain = pfrm;
	FrmSetActiveForm(pfrm);
	FrmSetEventHandler(pfrm, MainFormHandleEvent);
 	break;

      case frmOpenEvent:
	pfrm = FrmGetActiveForm();
	FrmDrawForm(pfrm);

	fill_form();
	break;

      case penDownEvent:
	switch(game_state) {
	case MESSAGE:
	  snd_clic();

	  /* remove dialog box (redraw everything) */
	  draw_level(marble_xp, marble_yp);
	  draw_sprite(MARBLE, marble_x>>8, marble_y>>8);
	  game_state=RUNNING;
	  break;

	case RUNNING:
	  /* collection bar */
	  if((e.screenY>143)&&(e.screenY<160))
	    if(x=get_from_collection(e.screenX>>4))
	      if((x&0xff)==DOCX)
		draw_message(x>>8);
	  break;

	case SETUP:
	  if(game_state==SETUP)	FrmDispatchEvent(&e);
	  break;
	}
	
	pen_last_x = e.screenX;
	pen_last_y = e.screenY;
	break;

      case penMoveEvent:
	if(game_state==RUNNING)
	  marble_push(e.screenX - pen_last_x, e.screenY - pen_last_y);

	pen_last_x = e.screenX;
	pen_last_y = e.screenY;
	break;

	/* tap menu soft key to stop game */
      case keyDownEvent:
	if(e.data.keyDown.chr==menuChr)
	  game_end(true);
	break;

      case appStopEvent:
	/* save high scores */
	PrefSetAppPreferences( CREATOR, 0, 1, &prefs, sizeof(prefs), true);

	if(game_state != SETUP) Win2SetMono();
	return 0;

      default:
      Dft:
	if(game_state==SETUP)
	  FrmDispatchEvent(&e);
      }

      /* do some kind of 50Hz timer event */
      if(TimGetTicks()>last_ticks) {
	if(game_state==RUNNING) {
	  do_animations();
	  move_marble(&x, &y);
	}

	/* game state may change in move_marble() */
	if(game_state==RUNNING) {
	  redraw_sprite(MARBLE, x, y);
	
	  if((level_time%100)==0)
	    draw_state();

	  level_time+=40;
	}

	last_ticks=TimGetTicks()+ticks_wait;
      }
    }
  }
  return 0;
}

