/*
 * Simple http (web) server.
 * Jim Rees University of Michigan
 * Mark Eichin <eichin@thok.org> The Herd Of Kittens
 * Original version April 1998
 */

#pragma pack(2)

#include <Common.h>
#include <System/SysAll.h>
#include <System/DataMgr.h>
#include <System/NetMgr.h>
#include <System/SysEvtMgr.h>
#include <UI/UIAll.h>

#include <System/Unix/unix_stdio.h>
#include <System/Unix/unix_stdlib.h>
#include <System/Unix/unix_string.h>
#include <System/Unix/sys_types.h>
#include <System/Unix/sys_socket.h>
#include <System/Unix/netinet_in.h>
#include <System/Unix/arpa_inet.h>
#include <System/Unix/netdb.h>

#include "httpd.h"
#include "resid.h"

static Err OpenNet(void);
static Err CloseNet(void);
static Boolean httpd(EventPtr event);

FieldPtr hostfield, urlfield, ansfield, statefield;
int netisopen, sfd = -1;

/* I'm not sure what these are doing here. */
Err errno, h_errno;
Long AppNetTimeout;
NetHostInfoBufType AppHostInfo;
NetServInfoBufType AppServInfo;
Word AppNetRefnum;

DWord
PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
    int err;

    if (cmd == sysAppLaunchCmdNormalLaunch) {
	err = StartApplication();
	if (err)
	    return err;
	EventLoop();
	StopApplication();
	return 0;
    } else
	return sysErrParamErr;
}

StartApplication()
{
    FrmGotoForm(formID_httpd);

    return 0;
}

static Err
OpenNet(void)
{
    Err err;
    Word ifErrs;
    struct sockaddr_in saddr;

    if (netisopen)
	return 0;

    err = SysLibFind("Net.lib", &AppNetRefnum);
    if (err) {
	pr("SysLibFind error");
	return err;
    }

    err = NetLibOpen(AppNetRefnum, &ifErrs);
    if ((err && err != netErrAlreadyOpen) || ifErrs) {
	pr("NetLibOpen error");
	return err;
    }

    err = 0;
    AppNetTimeout = SysTicksPerSecond() * 2;

    if ((sfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	err = errno;
	pr("socket error");
	goto done;
    }

    /* Bind it to the http port */

    bzero(&saddr, sizeof saddr);
    saddr.sin_family = AF_INET;
    saddr.sin_port = htons(PORT);
    saddr.sin_addr.s_addr = INADDR_ANY;

    if (bind(sfd, (struct sockaddr *) &saddr, sizeof saddr) < 0) {
	err = errno;
	close(sfd);
	pr("bind error");
	goto done;
    }

    /* Set the backlog */

    if (listen(sfd, 1) < 0) {
	err = errno;
	close(sfd);
	pr("listen error %d", err);
	goto done;
    }

    netisopen++;

 done:
    if (err)
	NetLibClose(AppNetRefnum, false);
    return err;
}

static Err
CloseNet(void)
{
    if (!netisopen)
	return 0;

    close(sfd);
    NetLibClose(AppNetRefnum, false);
    netisopen = 0;
    return 0;
}

StopApplication()
{
    CloseNet();
}

EventLoop()
{
    short err;
    int formID;
    FormPtr form;
    EventType event;

    do {
	EvtGetEvent(&event, 200);

	if (SysHandleEvent(&event))
	    continue;
	if (MenuHandleEvent((void *)0, &event, &err))
	    continue;

	if (event.eType == frmLoadEvent) {
	    formID = event.data.frmLoad.formID;
	    form = FrmInitForm(formID);
	    FrmSetActiveForm(form);
	    switch (formID) {
	    case formID_httpd:
		FrmSetEventHandler(form, (FormEventHandlerPtr) httpd);
		break;
	    }
	}
	FrmDispatchEvent(&event);
    } while (event.eType != appStopEvent);
}

static Boolean
httpd(EventPtr event)
{
    FormPtr form;
    int handled = 0;

    switch (event->eType) {
    case frmOpenEvent:
	form = FrmGetActiveForm();
	hostfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_host));
	urlfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_url));
	ansfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_ans));
	statefield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_state));
	FrmDrawForm(form);
	prfield("ready", statefield);
	handled = 1;
	break;

    case ctlSelectEvent:  // A control button was pressed and released.
	if (event->data.ctlEnter.controlID == buttonID_go) {
	    serve();
	    handled = 1;
	} else if (event->data.ctlEnter.controlID == buttonID_stop) {
	    /* nothing special */
	    handled = 1;
	}
	break;

    case menuEvent:
	switch (event->data.menu.itemID) {
	case menuitemID_about:
	    FrmAlert(alertID_about);
	    break;
	}
	handled = 1;
	break;

    }
    return handled;
}

serve()
{
    int fd = -1, len, n;
    char buf[96], cmd[12];
    static char host[20], url[20];
    struct sockaddr_in saddr;

    /* Init net */

    if (OpenNet())
	return;

    while (1) {
	/* Accept connections */

	prfield("waiting...", statefield);
	len = sizeof saddr;
	if ((fd = accept(sfd, (struct sockaddr *) &saddr, &len)) < 0) {
	    if (errno == netErrTimeout) {
		if (!EvtSysEventAvail(true))
		    continue;
	    } else
		pr("accept error %d", errno);
	    break;
	}

	/* I'm not sure this is necessary */
	EvtResetAutoOffTimer();

	/* I can't get inet_ntoa() to work */
	NetLibAddrINToA(AppNetRefnum, saddr.sin_addr.s_addr, host);
	prfield(host, hostfield);
	url[0] = '\0';
	prfield(url, urlfield);
	prfield("receiving...", statefield);

	if ((n = read(fd, buf, sizeof buf)) <= 0) {
	    pr("read error");
	    break;
	}
	buf[n-1] = '\0';
	/* I've seen this from lynx. */
	if (buf[n-2] == 0xd) buf[n-2] = 0;

	if (sscanf(buf, "%11s %19s", cmd, url) != 2) {
	    pr("bad http");
	    break;
	}

	if (strcmp(cmd, "GET")) {
	    pr("bad http cmd \"%s\"", cmd);
	    break;
	}

	prfield(url, urlfield);
	prfield("sending...", statefield);

	if (!strcmp(url, "/"))
	    send_root_index(fd);
	else if (!strcmp(url, "/date/"))
	    send_date_index(fd);
	else if (!strncmp(url, "/date/", strlen("/date/")))
	    send_date_file(fd, url);
	else if (!strcmp(url, "/memo/"))
	    send_memo_index(fd);
	else if (!strncmp(url, "/memo/", strlen("/memo/")))
	    send_memo_file(fd, url);
	else if (!strcmp(url, "/doc/"))
	    send_doc_index(fd);
	else if (!strncmp(url, "/doc/", strlen("/doc/")))
	    send_doc_file(fd, url);
	else if (!strcmp(url, "/debug/"))
	    send_debug_index(fd);
	else if (!strcmp(url, "/about/"))
	    send_about_index(fd);
	else if (!strcmp(url, "/httpd-icon.gif"))
	    send_icon(fd, httpd_icon, httpd_icon_size);
	else
	    send_404(fd, url);

	close(fd);
	fd = -1;
    }

    if (fd >= 0)
	close(fd);
    prfield("stopped", statefield);
}

char html0[] = "HTTP/1.0 200 OK\nMIME-version: 1.0\nContent-type: %s\n\n";

static char html_root[] =
    "<h1>Pilot Server</h1>\n"
    "<img width=32 height=20 src=\"httpd-icon.gif\"><p>\n"
    "Contents:<ul>\n"
    "<li> <a href=\"date/\">Datebook</a>\n"
    "<li> <a href=\"memo/\">Memos</a>\n"
    "<li> <a href=\"doc/\">Doc Files</a>\n"
    "<li> <a href=\"debug/\">Debug</a>\n"
    "<li> <a href=\"about/\">About httpd</a>\n"
    "</ul>\n"
;

char html_go_home[] =
    "<a href=\"/\"><img width=32 height=20 align=middle src=\"/httpd-icon.gif\"></a>\n"
    "Home.<p>\n"
;

int send_root_index(int fd)
{
    wrstr(fd, html0, "text/html");
    wrstr(fd, html_root);
}

static char about_text[] =
    "<h1>About httpd 1998-07-05</h1>\n"
    "Written by\n"
    "<a href=\"http://www.citi.umich.edu/u/rees/rees.html\">Jim Rees</a>\n"
    "and <a href=\"http://www.thok.org/~eichin\">Mark Eichin</a>.\n"
    "Original version April 1998. Current version and other goodies at Jim's\n"
    "<a href=\"http://www.citi.umich.edu/u/rees/pilot/\">PalmPilot Stuff</a>\n"
    "page.<p>\n"
;

int send_about_index(int fd)
{
    wrstr(fd, html0, "text/html");
    wrstr(fd, about_text);
    wrstr(fd, html_go_home);
}

send_icon(int fd, char *icon, int size)
{
    wrstr(fd, html0, "image/gif");
    write(fd, icon, size);
}

char html_notfound[] =
    "HTTP/1.0 404 Not Found\n"
    "Content-type: text/html\n\n"
    "<html><head><title>404 File Not Found</title></head>\n"
    "<body><h1>File Not Found</h1>\n"
    "%s not found.</body></html>\n"
;

send_404(int fd, char *url)
{
    wrstr(fd, html_notfound, (strlen(url) < 60) ? url : "[URL too long]");
}

int wrstr(int fd, const char *fmt, ...)
{
    va_list args;
    int i;
    char buf[400];

    va_start(args, fmt);
    i = vsprintf(buf, fmt, args);
    va_end(args);
    write(fd, buf, strlen(buf));
}

int wrs(int fd, char *s)
{
    write(fd, s, strlen(s));
}

int pr(const char *fmt, ...)
{
    va_list args;
    int i;
    static char buf[100];

    va_start(args, fmt);
    i = vsprintf(buf, fmt, args);
    va_end(args);
    prfield(buf, ansfield);
}

prfield(char *s, FieldPtr field)
{
    FldSetTextPtr(field, s);
    FldRecalculateField(field, true);
    FrmDrawForm(FrmGetActiveForm());
}

/*
copyright 1998
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works
and redistribute this software and such derivative works
for any purpose, so long as the name of the university of
michigan is not used in any advertising or publicity
pertaining to the use or distribution of this software
without specific, written prior authorization.  if the
above copyright notice or any other identification of the
university of michigan is included in any copy of any
portion of this software, then the disclaimer below must
also be included.

this software is provided as is, without representation
from the university of michigan as to its fitness for any
purpose, and without warranty by the university of
michigan of any kind, either express or implied, including
without limitation the implied warranties of
merchantability and fitness for a particular purpose. the
regents of the university of michigan shall not be liable
for any damages, including special, indirect, incidental, or
consequential damages, with respect to any claim arising
out of or in connection with the use of the software, even
if it has been or is hereafter advised of the possibility of
such damages.
*/
