/*************************************************************
 *
 * genpw.c:  Program to generate a random password for the
 *           Palm Connected Organizers.
 *
 * This program is distributed under the GNU General 
 * Public License (GPL).
 * See http://www.gnu.org/copyleft/gpl.html
 *
 * This software comes with NO warranty at all and I
 * cannot be held responsible for anything it may do.
 * I make no claims as to my ability to write programs.
 * You have the source code, so check it out.
 *
 * This program was developed on a Linux/GNU system using FREE
 * tools.  A special thanks to all programmers who developed
 * the following tools I used:
 *
 *   pilot-xfer-0.9.0
 *   gcc-2.7.2 cross compiler for Palm Connected Organizers
 *   gdb-4.16
 *   prctools-0.5.0
 *   pilot-template-1.31
 *   xcopilot-0.6.6
 *
 * Kevin Dupree
 * kdupree@flash.net
 *************************************************************/

#pragma pack(2)

#include "genpw.h"
#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>

/*************************************************************
 *      Defines                                              *
 *************************************************************/
#define genpwCreator   'GenP'
/* Define the minimum OS version allowed */
#define Version20    sysMakeROMVersion (2, 0, 0, 0, 0)
#define BITS_SET(w,b) (((w) & (b)) == (b))

/*************************************************************
 *	Global Variables                                     *
 *************************************************************/
/* Possible characters that can be used in password */
VoidHand hndValidChars;
/* Password used in Generation routine */
VoidHand hndPasswdChars;
/* Character version of number of characters in password */
VoidHand hndNumber;
/* Password used for display on form */
VoidHand hndPassword;
static CharPtr sLower = "abcdefghijklmnopqrstuvwxyz";
static CharPtr sUpper = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
static CharPtr sNumber = "1234567890";
static CharPtr sSpecial = "!@#$%^&*-_=+,<.>/?";
static Boolean bUseNumber = true;
static Boolean bUseSpecial = true;
static Boolean bUseUpper = true;
static Boolean bUseLower = true;
static Int iGenNumber = 10;


/*************************************************************
 *	Routines                                             *
 *************************************************************/
static CharPtr GeneratePW( void );
static VoidPtr Id2Ptr( Word );
static Word StartApplication (void);
static void StopApplication (void);
static Err CheckRomVersion(DWord requiredVersion, Word launchFlags);


/*************************************************************
 *	Type Definitions                                     *
 *************************************************************/

/* preferences state file. */
typedef struct {
  Boolean useUpper;
  Boolean useLower;
  Boolean useNumber;
  Boolean useSpecial;
  Int NumberChars;
} GenPWPrefType;

/*************************************************************
 *
 * NAME:        PilotMain
 *
 * DESCRIPTION: This is the main entry point for the GenPW
 *              application.
 *
 * REVISION HISTORY:
 *   Name   Date       Description
 *   ----   --------   -----------
 *   kld    5 Nov 98   Initial Revision
 *
 *************************************************************/
DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
  short err;
  EventType e;
  Word error;
  FormType *pfrm;
  FieldPtr fld;
  CharPtr sFld;
  ControlPtr ctl;

  err = CheckRomVersion (Version20, launchFlags);
  if (err) return (err);

  if (cmd == sysAppLaunchCmdNormalLaunch) 
    {
      error = StartApplication ();
      /* For now just use default settings if preferences load failed */
      /* if (error) return (error); */

      hndPassword = MemHandleNew ( 21 );
      if (hndPassword == NULL) return (1);
      hndValidChars = MemHandleNew ( 85 );
      if (hndValidChars == NULL) return(1);
      hndPasswdChars = MemHandleNew ( 21 );
      if (hndPasswdChars == NULL) return (1);
      hndNumber = MemHandleNew ( 3 );
      if (hndNumber == NULL) return (1);

      FrmGotoForm(genpwFormMain);

      while(1) 
        {
          EvtGetEvent(&e, 100);
          if (SysHandleEvent(&e)) 
            continue;
          if (MenuHandleEvent((void *)0, &e, &err)) 
            continue;
          
          switch (e.eType) 
            {
            case ctlSelectEvent:
              if (e.data.ctlSelect.controlID == idOKB)
                {
                  fld = Id2Ptr(idPasswdFld);
                  sFld = (CharPtr) MemHandleLock (hndPassword);
                  StrCopy( sFld, GeneratePW());
                  MemHandleUnlock (hndPassword);
                  FldSetTextHandle (fld, (Handle) hndPassword);
                  FldDrawField(fld);
                }
              else if (e.data.ctlSelect.controlID == idNumberCB)
                {
                  bUseNumber = !bUseNumber;
                }
              else if (e.data.ctlSelect.controlID == idSpecCharCB)
                {
                  bUseSpecial = !bUseSpecial;
                }
              else if (e.data.ctlSelect.controlID == idMixedPB)
                {
                  bUseUpper = true;
                  bUseLower = true;
                }
              else if (e.data.ctlSelect.controlID == idUpperPB)
                {
                  bUseUpper = true;
                  bUseLower = false;
                }
              else if (e.data.ctlSelect.controlID == idLowerPB)
                {
                  bUseUpper = false;
                  bUseLower = true;
                }
              else if (e.data.ctlSelect.controlID == idDownRB)
                {
                  if (iGenNumber > 4)
                    {
                      iGenNumber--;
                      fld = Id2Ptr(idNumberFld);
                      sFld = (CharPtr) MemHandleLock (hndNumber);
                      StrCopy ( sFld, "" );
                      StrIToA( sFld, iGenNumber);
                      MemHandleUnlock (hndNumber);
                      FldSetTextHandle (fld, (Handle) hndNumber);
                      FldDrawField(fld);
                    }
                }
              else if (e.data.ctlSelect.controlID == idUpRB)
                {
                  if (iGenNumber < 20)
                    {
                      iGenNumber++;
                      fld = Id2Ptr(idNumberFld);
                      sFld = (CharPtr) MemHandleLock (hndNumber);
                      StrCopy ( sFld, "" );
                      StrIToA( sFld, iGenNumber);
                      MemHandleUnlock (hndNumber);
                      FldSetTextHandle (fld, (Handle) hndNumber);
                      FldDrawField(fld);
                    }
                }
              break;
            case ctlRepeatEvent:
              
              if (e.data.ctlRepeat.controlID == idDownRB)
                {
                  if (iGenNumber > 4)
                    {
                      iGenNumber--;
                      fld = Id2Ptr(idNumberFld);
                      sFld = (CharPtr) MemHandleLock (hndNumber);
                      StrCopy ( sFld, "" );
                      StrIToA( sFld, iGenNumber);
                      MemHandleUnlock (hndNumber);
                      FldSetTextHandle (fld, (Handle) hndNumber);
                      FldDrawField(fld);
                    }
                  /* call FrmHandleEvent to finish processing for
                     repeating buttons */
                  FrmHandleEvent( FrmGetActiveForm(), &e );
                }
              else if (e.data.ctlRepeat.controlID == idUpRB)
                {
                  if (iGenNumber < 20)
                    {
                      iGenNumber++;
                      fld = Id2Ptr(idNumberFld);
                      sFld = (CharPtr) MemHandleLock (hndNumber);
                      StrCopy ( sFld, "" );
                      StrIToA( sFld, iGenNumber);
                      MemHandleUnlock (hndNumber);
                      FldSetTextHandle (fld, (Handle) hndNumber);
                      FldDrawField(fld);
                    }
                  /* call FrmHandleEvent to finish processing for
                     repeating buttons */
                  FrmHandleEvent( FrmGetActiveForm(), &e );
                }
              break;

            case frmLoadEvent:
              FrmSetActiveForm(FrmInitForm(e.data.frmLoad.formID));
              break;
	
            case frmOpenEvent:
              pfrm = FrmGetActiveForm();

              /* Display the selected number of characters */
              fld = Id2Ptr(idNumberFld);
              sFld = (CharPtr) MemHandleLock (hndNumber);
              StrCopy ( sFld, "" );
              StrIToA( sFld, iGenNumber);
              MemHandleUnlock (hndNumber);
              FldSetTextHandle (fld, (Handle) hndNumber);
              FrmDrawForm(pfrm);

              /* Select the Case push button */
              if (bUseLower == true && bUseUpper == true)
                {
                  fld = Id2Ptr(idMixedPB);
                  FrmSetControlGroupSelection( pfrm, 1, idMixedPB );
                }
              else if (bUseLower)
                {
                  fld = Id2Ptr(idLowerPB);
                  FrmSetControlGroupSelection( pfrm, 1, idLowerPB );
                }
              else
                {
                  fld = Id2Ptr(idUpperPB);
                  FrmSetControlGroupSelection( pfrm, 1, idUpperPB );
                }
              /* Intialize the Number Check Box */
              ctl = (ControlPtr) Id2Ptr (idNumberCB);
              CtlSetValue (ctl, bUseNumber);

              /* Initialize the Special Character Check Box */
              ctl = (ControlPtr) Id2Ptr (idSpecCharCB);
              CtlSetValue (ctl, bUseSpecial);

              break;

            case menuEvent:
              if (e.data.ctlSelect.controlID == idGenMenu)
                {
                  fld = Id2Ptr(idPasswdFld);
                  sFld = (CharPtr) MemHandleLock (hndPassword);
                  StrCopy( sFld, GeneratePW());
                  MemHandleUnlock (hndPassword);
                  FldSetTextHandle (fld, (Handle) hndPassword);
                  FldDrawField( fld );
                }
              else if (e.data.ctlSelect.controlID == idCopyMenu)
                {
                  /* Copy the current password to the clipboard */
                  fld = Id2Ptr( idPasswdFld );
                  FldSetSelection( fld, 0, iGenNumber );
                  FldCopy( fld );
                  FldSetSelection( fld, 0, 0 );
                }
              else if (e.data.ctlSelect.controlID == idHelpMenu)
                {
                  FrmHelp( genpwHelp1 );
                }
              else if (e.data.ctlSelect.controlID == idAboutMenu)
                {
                  FrmHelp( genpwAbout1 );
                }
              break;

            case appStopEvent:
              StopApplication();
              return 0;

            default:
              FrmHandleEvent( FrmGetActiveForm(), &e );

            }
        }

      /* Save the current preferences selected */
      StopApplication();
    }
  return 0;
}


/*************************************************************
 *
 * NAME:         StartApplication
 *
 * DESCRIPTION:  This routine loads the saved prefrences
 *               information and initializes global variables.
 *
 * REVISION HISTORY:
 *   Name   Date       Description
 *   ----   --------   -----------
 *   kld    5 Nov 98   Initial Revision
 *
 *************************************************************/
static Word StartApplication (void)
{
  Word error = 1;
  GenPWPrefType prefs;
  Int prefsVersion;
  Int prefsSize;

  /* Read the preferences saved-state information. */
  prefsSize = sizeof (GenPWPrefType);
  prefsVersion = PrefGetAppPreferences (genpwCreator,
                                        GenPWPrefID, 
                                        &prefs,
                                        &prefsSize,
                                        true);
  if (prefsVersion == GenPWPrefsVersionNum)
    {
      bUseUpper = prefs.useUpper;
      bUseLower = prefs.useLower;
      bUseNumber = prefs.useNumber;
      bUseSpecial = prefs.useSpecial;

      iGenNumber = prefs.NumberChars;
      error = 0;
    }
  /* Seed the random function */
  SysRandom(TimGetSeconds());

  return (error);
}


/*************************************************************
 *
 * NAME:        StopApplication
 *
 * DESCRIPTION: This routine saves the current state
 *              of the application.
 *
 * REVISION HISTORY:
 *   Name   Date       Description
 *   ----   --------   -----------
 *   kld    5 Nov 98   Initial Revision
 *
 *************************************************************/
static void StopApplication (void)
{
  GenPWPrefType prefs;

  prefs.useUpper = bUseUpper;
  prefs.useLower = bUseLower;
  prefs.useNumber = bUseNumber;
  prefs.useSpecial = bUseSpecial;
  prefs.NumberChars = iGenNumber;

  PrefSetAppPreferences (genpwCreator,
                         GenPWPrefID,
                         GenPWPrefsVersionNum,
                         &prefs, 
                         sizeof (GenPWPrefType),
                         true);

}


/*************************************************************
 *
 * NAME:        CheckRomVersion
 *
 * DESCRIPTION: This routine checks that a ROM version meets the
 *              minimum requirement.
 *
 * REVISION HISTORY:
 *   Name   Date       Description
 *   ----   --------   -----------
 *   kld    5 Nov 98   Initial Revision
 *
 *************************************************************/
static Err CheckRomVersion(DWord requiredVersion, Word launchFlags)
{
  DWord romVersion;

  /* Get the ROM version for what is running GenPW */
  FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
  if (romVersion >=  requiredVersion)
    return (0);

  
  /* If the user launched the app from the launcher, explain
     why the app shouldn't run using FrmAlert. */
  if (BITS_SET(launchFlags, sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp))
    {
      FrmAlert (RomIncompatibleAlert);
      if (romVersion < Version20)
        {
          Err err;     // Why is this var needed? See <AppLaunchCmd.h>!
          AppLaunchWithCommand(sysFileCDefaultApp,
                               sysAppLaunchCmdNormalLaunch,
                               NULL);
        }
    }

  return (sysErrRomIncompatible);
}


/*************************************************************
 *
 * NAME:        Id2Ptr
 *
 * DESCRIPTION: This routine is is used to convert an object ID 
 *              into a pointer to the object.
 *
 * REVISION HISTORY:
 *   Name   Date       Description
 *   ----   --------   -----------
 *   kld    5 Nov 98   Initial Revision
 *
 *************************************************************/
static VoidPtr Id2Ptr( Word wID )
{
  return( FrmGetObjectPtr( FrmGetActiveForm(), 
                           FrmGetObjectIndex( FrmGetActiveForm(), wID ) ) );
}


/*************************************************************
 *
 * NAME:        GeneratePW
 *
 * DESCRIPTION: This routine is used to generate
 *              a random password based on the user's selection
 *              for possible characters and requested size.
 *
 *
 * REVISION HISTORY:
 *   Name   Date       Description
 *   ----   --------   -----------
 *   kld    5 Nov 98   Initial Revision
 *
 *************************************************************/
static CharPtr GeneratePW( void )
{
  UInt NumberChars;
  CharPtr sValidChars;
  CharPtr sPassword;
  Char sTemp;
  int i;

  /* Build a string with all possible characters
     requested by the user */
  sValidChars = (CharPtr) MemHandleLock (hndValidChars);
  StrCopy(sValidChars, "");
  if ( bUseLower )
    StrCat( sValidChars, sLower);
  if ( bUseUpper )
    StrCat( sValidChars, sUpper);
  if ( bUseNumber )
    StrCat( sValidChars, sNumber);
  if ( bUseSpecial )
    StrCat( sValidChars, sSpecial);
  MemHandleUnlock (hndValidChars);

  /* Generate the password of the requested size */
  NumberChars = StrLen(sValidChars);
  sPassword = (CharPtr) MemHandleLock (hndPasswdChars);
  StrCopy(sPassword, "");
  MemHandleUnlock (hndPasswdChars);
  for (i=0; i<iGenNumber; i++) {
    sTemp = sValidChars[SysRandom(0)%NumberChars];
    sPassword = (CharPtr) MemHandleLock (hndPasswdChars);
    StrCat(sPassword, &sTemp);
    MemHandleUnlock (hndPasswdChars);
  }
  return sPassword;
}
