# $Id: LogLevel.pm,v 1.2 2007-07-12 09:53:10 mike Exp $

package Keystone::Resolver::LogLevel;

use strict;
use warnings;


=head1 NAME

Keystone::Resolver::LogLevel - logging levels for Keystone Resolver

=head1 SYNOPSIS

 $resolver->option(loglevel => (Keystone::Resolver::LogLevel::CHITCHAT &
                                Keystone::Resolver::LogLevel::CACHECHECK));
 print Keystone::Resolver::LogLevel::label(Keystone::Resolver::LogLevel::PARSEXSLT);

=head1 DESCRIPTION

This trivial class supplies a set of symbolic constants for the
bitmasks used in log-levels for Keystone Resolver logs.  These may be
arithmetrically ORred together to provide the resolver with a
requested logging level consisting of more than one of the levels
defined here.

=head1 LOGGING LEVELS

=over 4

=item CHITCHAT

Notify events of general interest such as starting up and shutting
down.  (DEADLY WARNING: Only use this if you like that kind of thing.)

=item CACHECHECK

Notify when checking for the existence of any kind of object in a
cache.

=item PARSEXSLT

Notify when parsing an XSLT stylesheet.

=item DUMPDESCRIPTORS

Dump the contents of the parsed descriptors.

=item SHOWGENRE

Dump the referent object before trying to resolve it.

=item SHOWGENRE

Show the genre (both ID and name) of each object resolved.

=item DBLOOKUP

Show the results of looking up IDs and tags in the resource database.

=item MKRESULT

Show the results of looking up IDs and tags in the resource database.

=item SQL

Show SQL statements before executing them.

=item DEREFERENCE

Indicate when an element of an OpenURL is dereferenced.  This includes
both a By-Reference ContextObject, and individual entities that are
specified by reference.

=item DISSECT

Indicate when a By-Value OpenURL (including one in which that value
was fetched because it was originally By-Reference) is dissected into
KEV elements.

=item RESOLVEID

Indicate when any kind of identifier (DOI, PubMed ID, etc.) is
resolved, and what the results are.

=item CONVERT01

Log the conversion of OpenURL v0.1 parameters to their v1.0
equivalents, and the creation of additional parameters required by
v1.0.

=item HANDLE

Indicate when a service handles a request, and whether the result is a
usable URL or a fatal or non-fatal error.

=item WARNING

Warnings generated by bad recipes and suchlike.

=back

=cut

sub CHITCHAT        {    0x1 }
sub CACHECHECK      {    0x2 }
sub PARSEXSLT       {    0x4 }
sub DUMPDESCRIPTORS {    0x8 }
sub DUMPREFERENT    {   0x10 }
sub SHOWGENRE       {   0x20 }
sub DBLOOKUP        {   0x40 }
sub MKRESULT        {   0x80 }
sub SQL             {  0x100 }
sub DEREFERENCE     {  0x200 }
sub DISSECT         {  0x400 }
sub RESOLVEID       {  0x800 }
sub CONVERT01       { 0x1000 }
sub HANDLE          { 0x2000 }
sub WARNING         { 0x4000 }


=head1 METHODS

=cut


=head2 label()

 print Keystone::Resolver::LogLevel::label(Resolver::LogLevel::PARSEXSLT);

Returns a short human-readable string describing the logging-level
specified as the argument.

=cut

sub label {
    my($level) = @_;

    if    ($level == CHITCHAT)        { return "CHITCHAT" }
    elsif ($level == CACHECHECK)      { return "CACHECHECK" }
    elsif ($level == PARSEXSLT)       { return "PARSEXSLT" }
    elsif ($level == DUMPDESCRIPTORS) { return "DUMPDESCRIPTORS" }
    elsif ($level == DUMPREFERENT)    { return "DUMPREFERENT" }
    elsif ($level == SHOWGENRE)       { return "SHOWGENRE" }
    elsif ($level == DBLOOKUP)        { return "DBLOOKUP" }
    elsif ($level == MKRESULT)        { return "MKRESULT" }
    elsif ($level == SQL)             { return "SQL" }
    elsif ($level == DEREFERENCE)     { return "DEREFERENCE" }
    elsif ($level == DISSECT)         { return "DISSECT" }
    elsif ($level == RESOLVEID)       { return "RESOLVEID" }
    elsif ($level == CONVERT01)       { return "CONVERT01" }
    elsif ($level == HANDLE)          { return "HANDLE" }
    elsif ($level == WARNING)         { return "WARNING" }

    return "[unknown LogLevel $level]";
}


1;
