package OpenInteract2::Brick::BaseSecurity;

use strict;
use base qw( OpenInteract2::Brick );
use OpenInteract2::Exception;

my %INLINED_SUBS = (
    'base_security-2.22.zip' => 'BASE_SECURITY222ZIP',
);

sub get_name {
    return 'base_security';
}

sub get_resources {
    return (
        'base_security-2.22.zip' => [ 'pkg base_security-2.22.zip', 'no' ],
    );
}

sub load {
    my ( $self, $resource_name ) = @_;
    my $inline_sub_name = $INLINED_SUBS{ $resource_name };
    unless ( $inline_sub_name ) {
        OpenInteract2::Exception->throw(
            "Resource name '$resource_name' not found ",
            "in ", ref( $self ), "; cannot load content." );
    }
    return $self->$inline_sub_name();
}

OpenInteract2::Brick->register_factory_type( get_name() => __PACKAGE__ );

=pod

=head1 NAME

OpenInteract2::Brick::BaseSecurity - Base-64 encoded OI2 package 'base_security-2.22.zip' shipped with distribution

=head1 SYNOPSIS

  oi2_manage create_website --website_dir=/path/to/site

=head1 DESCRIPTION

Are you sure you even need to be reading this? If you are just looking
to install a package just follow the instructions from the SYNOPSIS.

Still here? This class holds the Base64-encoded versions of package
file "base_security-2.22.zip" shipped with OpenInteract2. Once you decode them you
should store them as a ZIP file and then read them in with
Archive::Zip or some other utility.

A typical means to do this is:

 my $brick = OpenInteract2::Brick->new( 'base_security' );

 # there's only one resource in this brick...
 my ( $pkg_name ) = $brick->list_resources;
 my $pkg_info = $brick->load_resource( $pkg_name );
 my $pkg_file = OpenInteract2::Util->decode_base64_and_store(
     \$pkg_info->{content}
 );

 # $pkg_file now references a file on the filesystem that you can
 # manipulate as normal

These resources are associated with OpenInteract2 version 1.99_06.


=head2 Resources

You can grab resources individually using the names below and
C<load_resource()> and C<copy_resources_to()>, or you can copy all the
resources at once using C<copy_all_resources_to()> -- see
L<OpenInteract2::Brick> for details.

=over 4


=item B<base_security-2.22.zip>


=back

=head1 COPYRIGHT

Copyright (c) 2005 Chris Winters. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS


Chris Winters E<lt>chris@cwinters.comE<gt>


=cut


sub BASE_SECURITY222ZIP {
    return <<'SOMELONGSTRING';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SOMELONGSTRING
}

